package Admin::SkinControl;
use strict;
#################################################################################
# Ikonboard v3 by Jarvis Entertainment Group, Inc.
#
# No part of this script can be used outside Ikonboard without prior consent.
#
# More information available from <ib-license@jarvisgroup.net>
# (c)2001 Jarvis Entertainment Group, Inc.
# 
# http://www.ikonboard.com | http://www.jarvisgroup.net
#
# Please read the license for more information
#
# 
# Script Author: Matthew Mecham <matt@ikonboard.com>
#
#################################################################################

BEGIN {
    require 'Lib/FUNC.pm';
    require 'Lib/ADMIN.pm';
    require 'Admin/SKIN.pm';
    require 'Boardinfo.pm';
}


my $SKIN  = Admin::SKIN->new();
my $std   = FUNC::STD->new();
my $ADMIN = FUNC::ADMIN->new();
my $INFO  = Boardinfo->new();

sub new {
    my $pkg = shift;
    my $obj = {};
    bless $obj, $pkg;
    return $obj;
}

#########################################################################################################
#
# Splash
#
#########################################################################################################


sub splash {
    my ($obj, $db) = @_;

    my @editable; 
    my @skins = split( /\|\&\|/, $INFO->{'SKINS'} );
    for my $s (@skins) {
        @_ = split /\:/, $s;
        push @editable, { NAME => $_[2], VALUE => $_[1] };
    }

    my $size = scalar @editable + 1;

    my $html  = $SKIN->title( TITLE => 'Styles/Skin Control' );
       $html .= $SKIN->begin_table();
       $html .= $SKIN->form_start();
       $html .= $SKIN->hidden_fields( { act   => 'styles',
                                        CODE  => 'edit'
                                      } );

    #+-----------------------------------

    $html .= $SKIN->section_header( TITLE => "Please Choose 'Skin' to edit" );

    $html .= $SKIN->td_select( TEXT     => 'Installed Skins',
                               NAME     => 'SKIN',
                               SIZE     => $size,
                               VALUES   => 'Default',
                               DATA     => \@editable,
                             );

    $html .= $SKIN->section_header( TITLE => "Skin Sections" );

    $html .= $SKIN->td_select( TEXT     => 'Skin Section',
                               NAME     => 'SECTION',
                               SIZE     => 6,
                               VALUES   => '',
                               DATA     => [ { NAME => 'CSS (Style Sheets)',   VALUE => 'CSS'   },
                                             { NAME => 'Javascript',           VALUE => 'JS'    },
                                             { NAME => 'Global Colours',       VALUE => 'COLOR' },
                                             { NAME => 'Icon/GFX Attributes',  VALUE => 'GFX'   },
                                             { NAME => 'HTML',                 VALUE => 'HTML'  },
                                             { NAME => 'Apply Board Template', VALUE => 'TEMP'  },
                                           ]
                             );




    $html .= $SKIN->td_submit(   NAME => '', VALUE => 'Edit this Skin' );


    $html .= $SKIN->form_end();
    $html .= $SKIN->end_table();

    #+-----------------------------------

    $ADMIN->Output( WHERE => 'STYLES', NAV_ONE => 'Styles/Skin Control', PRINT => $html);

}

#########################################################################################################
#
# Edit / Do Edit Jumpers
#
#########################################################################################################

sub edit {
    my ($obj, $db) = @_;

    $ADMIN->Error( DB=>,"", STD=>"", MSG => "You must specify which skin to edit")    unless defined $iB::IN{'SKIN'};
    $ADMIN->Error( DB=>,"", STD=>"", MSG => "You must specify which section to edit") unless defined $iB::IN{'SECTION'};

    my %ist_gut = ( 'CSS'    => \&CSS,
                    'JS'     => \&JS,
                    'COLOR'  => \&COLOR,
                    'GFX'    => \&GFX,
                    'HTML'   => \&HTML,
                    'TEMP'   => \&TEMP,
                  );

    $ist_gut{ $iB::IN{'SECTION'} }->($obj, $db); #ja?
}

sub doedit {
    my ($obj, $db) = @_;


    $ADMIN->Error( DB=>,"", STD=>"", MSG => "You must specify which skin to edit")    unless defined $iB::IN{'SKIN'};
    $ADMIN->Error( DB=>,"", STD=>"", MSG => "You must specify which section to edit") unless defined $iB::IN{'SECTION'};

    my %ist_gut = ( 'CSS'   => \&do_CSS,
                    'JS'    => \&do_JS,
                    'COLOR' => \&do_COLOR,
                    'GFX'   => \&do_GFX,
                    'HTML'  => \&do_HTML,
                    'TEMP'  => \&do_TEMP,
                  );

    $ist_gut{ $iB::IN{'SECTION'} }->($obj, $db); #ja?
}

#########################################################################################################
#
# Apply TEMPLATE Routines
#
#########################################################################################################

sub TEMP {
    my ($obj, $db) = @_;

    my $html  = $SKIN->title( TITLE => 'Template Control');
       $html .= $SKIN->begin_table();
       $html .= $SKIN->form_start();
       $html .= $SKIN->hidden_fields( { act     => 'styles',
                                        CODE    => 'doedit',
                                        SECTION => 'TEMP',
                                        SKIN    => $iB::IN{'SKIN'}
                                      } );

    #+-----------------------------------

    # Get the templates...
    my $templates = $db->query( TABLE  => 'templates' );
    my @rtemplates;

    for (@{$templates}) {
        next if $_->{'ID'} eq 'register';
        if ($_->{'ID'} eq 'global') {
            $_->{'NAME'} = 'Original Board Template';
        }
        push @rtemplates, { NAME => $_->{'NAME'}, VALUE => $_->{'ID'} };
    }

    # Do we already have a template?
    my $current;

    if ($INFO->{'SKIN_TEMPLATES'} =~ /$iB::IN{'SKIN'}\:(.+?)\|\&\|/) {
        $current = $1;
    }

    $html .= $SKIN->section_header( TITLE => "Please Choose a Template to apply to this skin" );

    $html .= $SKIN->td_select( TEXT     => 'Editable Templates',
                               NAME     => 'ID',
                               VALUES   => $current,
                               DATA     => \@rtemplates,
                             );

    $html .= $SKIN->td_submit(   NAME => '', VALUE => 'Apply this Template' );


    $html .= $SKIN->form_end();
    $html .= $SKIN->end_table();

    #+-----------------------------------

    $ADMIN->Output( WHERE => 'STYLES', NAV_ONE => 'Styles/Skin Control', PRINT => $html);

}


sub do_TEMP {
    my ($obj, $db) = @_;

    $ADMIN->Error( DB=>,"", STD=>"", MSG => "No template chosen!")  unless $iB::CGI->param('ID');

    my $final_skins;

    my $OLD = Boardinfo->new();
    # Remove the old template ID (if any)
    for my $sk ( split /\|\&\|/, $INFO->{'SKIN_TEMPLATES'} ) {
        my ($skin, $id) = split /\:/, $sk;
        unless ( $skin eq $iB::IN{'SKIN'} ) {
            $final_skins .= $skin.':'.$id.'|&|';
        }
    }
    # Add in the new..
    $final_skins .= $iB::IN{'SKIN'}.':'.$iB::IN{'ID'}.'|&|';

    $OLD->{'SKIN_TEMPLATES'} = $final_skins;
    # Update the Boardinfo.pm file
    $ADMIN->make_module( FILE     => "Boardinfo.pm",
                         PKG_NAME => 'Boardinfo',
                         VALUES   => $OLD
                       );

    $ADMIN->static_screen( URL   => "act=styles&CODE=edit&SECTION=$iB::IN{'SECTION'}&SKIN=$iB::IN{'SKIN'}",
                           TITLE => "Board Template Assigned",
                           TEXT  => "The changes were successful"
                         );
}


#########################################################################################################
#
# CSS Routines
#
#########################################################################################################


sub CSS {
    my ($obj, $db) = @_;

    my $file = $iB::INFO->{'HTML_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/ikonboard.css';

    unless (-e $file) {
        $ADMIN->Error( DB=>,"", STD=>"", MSG => "Cannot locate the CSS file at ".$file);
    }
   
    if ($^O ne 'MacOS' && ($^O ne 'MSWin32' || !Win32::IsWin95())) {
        unless (-w $file) {
            chmod (0777, $file) || $ADMIN->Error( DB=>"",STD=>"",MSG=>"I do not have write permissions to change the CHMOD value on :$file. Please use FTP to change the CHMOD value on the file $file");
            chmod (0777, $iB::INFO->{'HTML_DIR'}.'Skin/'.$iB::IN{'SKIN'}) || $ADMIN->Error( DB=>"",STD=>"",MSG=>"I do not have write permissions to change the CHMOD value on the path :$file. Please use FTP to change the CHMOD value on skins directory");
        }
    } else {
        unless (-w $file) {
            $ADMIN->Error( DB=>"",STD=>"",MSG=>"I do not have write permissions to $file");
        }
    }
    
    my $css;
    {
        local $/ = undef;
        open CSS, "<".$file or die $!;
        $css = <CSS>;
        close CSS;
    }

    my $html  = $SKIN->title( TITLE => 'Editing CSS', TEXT => "Partial Skin Support for BETA" );
       $html .= $SKIN->begin_table();
       $html .= $SKIN->form_start();
       $html .= $SKIN->hidden_fields( { act     => 'styles',
                                        CODE    => 'doedit',
                                        SECTION => 'CSS',
                                        SKIN    => $iB::IN{'SKIN'}
                                      } );

    #+-----------------------------------

    $html .= $SKIN->section_header( TITLE => "Edit the CSS file" );

    $html .= $SKIN->td_textarea( TEXT  => "CSS Information",
                                 NAME  => 'CSS_INFO',
                                 VALUE => $css,
                                 SPAN  => 1,
                                 ROWS  => 30
                                );
    $html .= $SKIN->td_submit(   NAME => '', VALUE => 'Apply this Edit' );


    $html .= $SKIN->form_end();
    $html .= $SKIN->end_table();

    #+-----------------------------------

    $ADMIN->Output( WHERE => 'STYLES', NAV_ONE => 'Styles/Skin Control', PRINT => $html);

}


#+----------------------------------------------------------------------------------------

sub do_CSS {
    my ($obj, $db) = @_;
    $ADMIN->Error( DB=>,"", STD=>"", MSG => "No CSS Information to save")  unless $iB::CGI->param('CSS_INFO');

    my $cfile = $iB::INFO->{'HTML_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/ikonboard.css';

    if ($^O eq 'MSWin32') {
        $cfile =~ s!/!\\!g;
    }
    unless (-e $cfile) {
        $ADMIN->Error( DB=>,"", STD=>"", MSG => "I cannot find a file called $cfile :(  .. Is it there? are your paths set up correctly?");
    }
    unless (-w $cfile) {
        $ADMIN->Error( DB=>,"", STD=>"", MSG => "I do not have permission to write to $cfile :( ... Is the CHMOD value correct?");
    }
    
    my $info = $iB::CGI->param('CSS_INFO');

    if (open (NETSCAPESUCKS, ">".$cfile)) {
        print NETSCAPESUCKS $info;
        close NETSCAPESUCKS;
    } else {
         $ADMIN->Error( DB=>,"", STD=>"", MSG => "Opening $cfile failed. Why? I don't know: Maybe perl does (\$! = $1 and \$@ = $@)");
    }


    $ADMIN->static_screen( URL   => "act=styles&CODE=edit&SECTION=$iB::IN{'SECTION'}&SKIN=$iB::IN{'SKIN'}",
                           TITLE => "CSS Information edited",
                           TEXT  => "The changes were successful"
                         );
}

#########################################################################################################
#
# Javascript routines
#
#########################################################################################################



sub JS {
    my ($obj, $db) = @_;

    my $file = $iB::INFO->{'HTML_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/ikonboard.js';

    unless (-e $file) {
        $ADMIN->Error( DB=>,"", STD=>"", MSG => "Cannot locate the CSS file at ".$file);
    }
   
    if ($^O ne 'MacOS' && ($^O ne 'MSWin32' || !Win32::IsWin95())) {
        unless (-w $file) {
            chmod (0777, $file) || $ADMIN->Error( DB=>"",STD=>"",MSG=>"I do not have write permissions to change the CHMOD value on :$file. Please use FTP to change the CHMOD value on the file $file");
            chmod (0777, $iB::INFO->{'HTML_DIR'}.'Skin/'.$iB::IN{'SKIN'}) || $ADMIN->Error( DB=>"",STD=>"",MSG=>"I do not have write permissions to change the CHMOD value on the path :$file. Please use FTP to change the CHMOD value on skins directory");
        }
    } else {
        unless (-w $file) {
            $ADMIN->Error( DB=>"",STD=>"",MSG=>"I do not have write permissions to $file");
        }
    }

    local $/ = undef;
    open JAVA, "<".$file or die $!;
    my $js = <JAVA>;
    close JAVA;
    my $html  = $SKIN->title( TITLE => 'Editing iB Javascript', TEXT => "Partial Skin Support for BETA" );
       $html .= $SKIN->begin_table();
       $html .= $SKIN->form_start();
       $html .= $SKIN->hidden_fields( { act     => 'styles',
                                        CODE    => 'doedit',
                                        SECTION => 'JS',
                                        SKIN    => $iB::IN{'SKIN'}
                                      } );

    #+-----------------------------------

    $html .= $SKIN->section_header( TITLE => "Edit the JS file" );

    $html .= $SKIN->td_textarea( TEXT  => "JS Information",
                                 NAME  => 'JS_INFO',
                                 VALUE => $js,
                                 SPAN  => 1,
                                 ROWS  => 30
                                );
    $html .= $SKIN->td_submit(   NAME => '', VALUE => 'Apply this Edit' );


    $html .= $SKIN->form_end();
    $html .= $SKIN->end_table();

    #+-----------------------------------

    $ADMIN->Output( WHERE => 'STYLES', NAV_ONE => 'Styles/Skin Control', PRINT => $html);

}

#+----------------------------------------------------------------------------------------

sub do_JS {
    my ($obj, $db) = @_;

    $ADMIN->Error( DB=>,"", STD=>"", MSG => "No JS Information to save")  unless $iB::CGI->param('JS_INFO');

    my $file = $iB::INFO->{'HTML_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/ikonboard.js';

    open  JFILE, ">".$file or die $!;
    print JFILE $iB::CGI->param('JS_INFO');
    close JFILE;

    $ADMIN->write_log( TITLE => 'Javascript Edited');

    $ADMIN->static_screen( URL   => "act=styles&CODE=edit&SECTION=$iB::IN{'SECTION'}&SKIN=$iB::IN{'SKIN'}",
                           TITLE => "Javascript Information edited",
                           TEXT  => "The changes were successful"
                         );
}

#########################################################################################################
#
# Global Colours routines
#
#########################################################################################################

sub COLOR {
    my ($obj, $db) = @_;

    require $iB::INFO->{'IKON_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/Styles.pm';

    my $r_skin = Styles->new();

    $r_skin->{'EDIT_FONT_SIZE'} ||= 7;


    my $html = qq~

                 <script language="javascript">
                 <!--
                 function PopUp(url, name, width,height,center,resize,scroll,posleft,postop) {
                    if (posleft != 0) { x = posleft }
                    if (postop  != 0) { y = postop  }

                    if (!scroll) { scroll = 1 }
                    if (!resize) { resize = 1 }

                    if ((parseInt (navigator.appVersion) >= 4 ) && (center)) {
                        X = (screen.width  - width ) / 2;
                        Y = (screen.height - height) / 2;
                    }
                    if (scroll != 0) { scroll = 1 }
                    var Win = window.open( url, name, 'width='+width+',height='+height+',top='+Y+',left='+X+',resizable='+resize+',scrollbars='+scroll+',location=no,directories=no,status=no,menubar=no,toolbar=no');
                  }
                //-->
                </script>
                ~;

       $html .= $SKIN->title( TITLE => 'Editing Global Colours', TEXT => "" );
       $html .= $SKIN->begin_table();
       $html .= $SKIN->form_start();
       $html .= $SKIN->hidden_fields( { act     => 'styles',
                                        CODE    => 'doedit',
                                        SECTION => 'COLOR',
                                        SKIN    => $iB::IN{'SKIN'}
                                      } );

    $html .= $SKIN->section_header( TITLE => "Need a colour chart reference?", TEXT => qq!<a href="javascript:PopUp('$INFO->{IMAGES_URL}/sys-img/colours.html','Colours','500','300','0','1','1','1')">Click here (Launches a new window)</a>! );

    $html .= $SKIN->section_header( TITLE => "Table Information" );

    $html .= $SKIN->td_input ( TEXT  => 'Table Width',
                               NAME  => 'TABLE_WIDTH',
                               VALUE => $r_skin->{'TABLE_WIDTH'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Table Border Colour',
                               NAME  => 'TABLE_BORDER_COL',
                               VALUE => $r_skin->{'TABLE_BORDER_COL'},
                               REQ   => 1
                             );

    $html .= $SKIN->section_header( TITLE => "Logo Information", TEXT=>"Please make sure the logo is in the Images directory in the skin directory you are editing. There is no need to add the full URL");

    $html .= $SKIN->td_input ( TEXT  => 'Board Logo',
                               NAME  => 'BOARD_LOGO',
                               VALUE => $r_skin->{'BOARD_LOGO'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Logo Alignment',
                               NAME  => 'LOGO_ALIGN',
                               VALUE => $r_skin->{'LOGO_ALIGN'},
                               REQ   => 1
                             );

    $html .= $SKIN->section_header( TITLE => "Miscellaneous Colours", TEXT => "For posting screens, logging in screens, registration screens, etc");

    $html .= $SKIN->td_input ( TEXT  => 'Board HTML Background Colour',
                               NAME  => 'HTML_BACK_COL',
                               VALUE => $r_skin->{'HTML_BACK_COL'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Misc. Table Cell Colour #1 (Light)',
                               NAME  => 'MISCBACK_ONE',
                               VALUE => $r_skin->{'MISCBACK_ONE'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Misc. Table Cell Colour #2 (Dark)',
                               NAME  => 'MISCBACK_TWO',
                               VALUE => $r_skin->{'MISCBACK_TWO'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Misc. Header Table Cell Colour',
                               NAME  => 'MISCBACK_TITLE',
                               VALUE => $r_skin->{'MISCBACK_TITLE'},
                               REQ   => 1
                             );

    $html .= $SKIN->section_header( TITLE => "Table Title Colours", TEXT => "For posting screens, logging in screens, registration screens, etc");

    $html .= $SKIN->td_input ( TEXT  => 'Title Background Colour',
                               NAME  => 'TITLEBACK',
                               VALUE => $r_skin->{'TITLEBACK'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Title Table Border Colour',
                               NAME  => 'TITLEBORDER',
                               VALUE => $r_skin->{'TITLEBORDER'},
                               REQ   => 1
                             );

    $html .= $SKIN->section_header( TITLE => "Forum Column Colours");

    $html .= $SKIN->td_input ( TEXT  => 'Forum Table Cell Colour #1',
                               NAME  => 'FORUM_COL_ONE',
                               VALUE => $r_skin->{'FORUM_COL_ONE'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Forum Table Cell Colour #2',
                               NAME  => 'FORUM_COL_TWO',
                               VALUE => $r_skin->{'FORUM_COL_TWO'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Pinned Topic Table Cell Colour #1',
                               NAME  => 'PIN_COL_ONE',
                               VALUE => $r_skin->{'PIN_COL_ONE'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Pinned Topic Table Cell Colour #2',
                               NAME  => 'PIN_COL_TWO',
                               VALUE => $r_skin->{'PIN_COL_TWO'},
                               REQ   => 1
                             );

    $html .= $SKIN->section_header( TITLE => "Category Colours");

    $html .= $SKIN->td_input ( TEXT  => 'Category Title Background Colour',
                               NAME  => 'CAT_BACK',
                               VALUE => $r_skin->{'CAT_BACK'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Nested Category (Expanded) Col. #1',
                               NAME  => 'EXP_CAT_ONE',
                               VALUE => $r_skin->{'EXP_CAT_ONE'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Nested Category (Expanded) Col. #2',
                               NAME  => 'EXP_CAT_TWO',
                               VALUE => $r_skin->{'EXP_CAT_TWO'},
                               REQ   => 1
                             );

    $html .= $SKIN->section_header( TITLE => "Thread View Row Colours");

    $html .= $SKIN->td_input ( TEXT  => 'Thread Entry Colour #1',
                               NAME  => 'POST_COL_ONE',
                               VALUE => $r_skin->{'POST_COL_ONE'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Thread Entry Colour #2',
                               NAME  => 'POST_COL_TWO',
                               VALUE => $r_skin->{'POST_COL_TWO'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Edited by font size (in pt)',
                               NAME  => 'EDIT_FONT_SIZE',
                               VALUE => $r_skin->{'EDIT_FONT_SIZE'},
                               REQ   => 1
                             );

    $html .= $SKIN->section_header( TITLE => "User CP, User Bar Colours");

    $html .= $SKIN->td_input ( TEXT  => 'User Strip Background Colour',
                               NAME  => 'USER_BAR_BACK',
                               VALUE => $r_skin->{'USER_BAR_BACK'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'User CP Navigation Tabs: Active',
                               NAME  => 'USERNAV_ON',
                               VALUE => $r_skin->{'USERNAV_ON'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'User CP Navigation Tabs: Inactive',
                               NAME  => 'USERNAV_OFF',
                               VALUE => $r_skin->{'USERNAV_OFF'},
                               REQ   => 1
                             );

    $html .= $SKIN->section_header( TITLE => "Navigation Strip Colours");

    $html .= $SKIN->td_input ( TEXT  => 'Navigation Border Colour',
                               NAME  => 'NAV_BORDER',
                               VALUE => $r_skin->{'NAV_BORDER'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Navigation Table Background Colour',
                               NAME  => 'NAV_BACK',
                               VALUE => $r_skin->{'NAV_BACK'},
                               REQ   => 1
                             );
#################
# Added by Infection
    $html .= $SKIN->section_header( TITLE => "Calendar Colours");

    $html .= $SKIN->td_input ( TEXT  => 'Blank cell',
                               NAME  => 'CALENDAR_BLANK',
                               VALUE => $r_skin->{'CALENDAR_BLANK'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Cells with old or new days',
                               NAME  => 'CALENDAR_NON',
                               VALUE => $r_skin->{'CALENDAR_NON'},
                               REQ   => 1
                             );

    $html .= $SKIN->td_input ( TEXT  => 'Current day cell',
                               NAME  => 'CALENDAR_CUR',
                               VALUE => $r_skin->{'CALENDAR_CUR'},
                               REQ   => 1
                             );
#
#######


    $html .= $SKIN->td_submit(   NAME => '', VALUE => 'Apply this Edit' );


    $html .= $SKIN->form_end();
    $html .= $SKIN->end_table();

    #+-----------------------------------

    $ADMIN->Output( WHERE => 'STYLES', NAV_ONE => 'Styles/Skin Control', PRINT => $html);

}

sub do_COLOR {
    my ($obj, $db) = @_;

    do "$iB::IN{'SKIN'}/Styles.pm" or die $!;

    my $OLD = Styles->new();

    my $data = {    HTML_BACK_COL   => $iB::IN{'HTML_BACK_COL'},
                    TITLEBACK       => $iB::IN{'TITLEBACK'},
                    TITLEBORDER     => $iB::IN{'TITLEBORDER'},
                    USER_BAR_BACK   => $iB::IN{'USER_BAR_BACK'},
                    NAV_BORDER      => $iB::IN{'NAV_BORDER'},
                    NAV_BACK        => $iB::IN{'NAV_BACK'},
                    USERNAV_OFF     => $iB::IN{'USERNAV_OFF'},
                    USERNAV_ON      => $iB::IN{'USERNAV_ON'},
                    MISCBACK_ONE    => $iB::IN{'MISCBACK_ONE'},
                    MISCBACK_TWO    => $iB::IN{'MISCBACK_TWO'},
                    MISCBACK_TITLE  => $iB::IN{'MISCBACK_TITLE'},
                    CAT_BACK        => $iB::IN{'CAT_BACK'},
                    TABLE_BORDER_COL=> $iB::IN{'TABLE_BORDER_COL'},
                    TABLE_WIDTH     => $iB::IN{'TABLE_WIDTH'},
                    FORUM_COL_ONE   => $iB::IN{'FORUM_COL_ONE'},
                    FORUM_COL_TWO   => $iB::IN{'FORUM_COL_TWO'},
                    PIN_COL_ONE     => $iB::IN{'PIN_COL_ONE'},
                    PIN_COL_TWO     => $iB::IN{'PIN_COL_TWO'},
                    EXP_CAT_ONE     => $iB::IN{'EXP_CAT_ONE'},
                    EXP_CAT_TWO     => $iB::IN{'EXP_CAT_TWO'},
                    POST_COL_ONE    => $iB::IN{'POST_COL_ONE'},
                    POST_COL_TWO    => $iB::IN{'POST_COL_TWO'},
                    BOARD_LOGO      => $iB::IN{'BOARD_LOGO'},
                    LOGO_ALIGN      => $iB::IN{'LOGO_ALIGN'},
                    EDIT_FONT_SIZE  => $iB::IN{'EDIT_FONT_SIZE'},
######
# adde by Infection
                    CALENDAR_BLANK  => $iB::IN{'CALENDAR_BLANK'},
                    CALENDAR_NON    => $iB::IN{'CALENDAR_NON'},
                    CALENDAR_CUR    => $iB::IN{'CALENDAR_CUR'},
#
#####
                };

    for my $k (keys %{$data}) {
        $OLD->{$k} = $data->{$k};
    }

    # We need to make safe our variables
    # so that they are not interpolated when
    # saved back
    for my $v (keys %{$OLD}) {
        $OLD->{$v} =~ s!$iB::INFO->{'IMAGES_URL'}!\$iB::INFO->{'IMAGES_URL'}!g;
    }

    $ADMIN->make_module( FILE     => 'Styles.pm',
                         PKG_NAME => 'Styles',
                         VALUES   => $OLD,
                         INTERPOLATE => "yes",
                         PATH     => $iB::INFO->{'IKON_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/'
                       );

    $ADMIN->write_log( TITLE => 'Global Colours Edited');

    $ADMIN->static_screen( URL   => "act=styles&CODE=edit&SECTION=$iB::IN{'SECTION'}&SKIN=$iB::IN{'SKIN'}",
                           TITLE => "Colour Information edited",
                           TEXT  => "The changes were successful"
                         );
}

#########################################################################################################
#
# GFX Attributes routines
#
#########################################################################################################

sub GFX {
    my ($obj, $db) = @_;

    $ADMIN->Error( DB=>,"", STD=>"", MSG => "Not a valid skin package") unless $iB::IN{'SKIN'};

    require $iB::IN{'SKIN'} . '/Styles.pm' or die $!;
    do $iB::INFO->{'IKON_DIR'} . 'Skin/' . $iB::IN{'SKIN'} . '/gfx_data.cfg' or die $!;

    my $GFX = gfx_data->new();

    my $html .= $SKIN->title( TITLE => 'Editing Graphics Properties', TEXT => "If you choose the Type property as a 'Flash Movie', the relative URL must end in '.swf'. If you choose the Property Type as 'Text' then you must enter the text into the GFX Reference box - you can omit the width, height and alt properties.<br><br>When using 'Graphic' or 'Flash Movie', the image/movie MUST be uploaded into the relative 'iB_html/non-cgi/Skin/\$skin_name/images' directory and enter the graphic/movie name only (ie: piccie.gif, movie.swf) into the GFX Reference box" );
    $html .= $SKIN->begin_table();
    $html .= $SKIN->form_start();
    $html .= $SKIN->hidden_fields( {    act     => 'styles',
                                        CODE    => 'doedit',
                                        SECTION => 'GFX',
                                        SKIN    => $iB::IN{'SKIN'}
                                      } );

    $html .= $SKIN->section_header( TITLE => "Your Ikonboard Graphic Properties");

    $html .= qq~<tr><td colspan='2'>
                <table width='100%' border='0' class='t' cellspacing='1' cellpadding='5'>
                <tr>
                  <td bgcolor='#8888AA' width='20%'><b>GFX Reference</b></td>
                  <td bgcolor='#8888AA' width='20%'><b>Link Property</b></td>
                  <td bgcolor='#8888AA' width='5%'><b>Width Attribute</b></td>     
                  <td bgcolor='#8888AA' width='5%'><b>Height Attribute</b></td> 
                  <td bgcolor='#8888AA' width='15%'><b>ALT Attribute</b></td>
                  <td bgcolor='#8888AA' width='5%'><b>Border Width</b></td>
                  <td bgcolor='#8888AA' width='15%'><b>Align Attribute</b></td>
                </tr>
                ~;

    for my $g (sort keys %{$GFX}) {

        my $the_options = qq~<option value=''>NONE</option><option value='bottom'>bottom</option><option value='center'>center</option><option value='middle'>middle</option><option value='absmiddle'>absmiddle</option><option value='absbottom'>absbottom</option><option value='top'>top</option><option value='left'>left</option><option value='right'>right</option>~;

        $the_options =~ s!value='$GFX->{$g}[7]'>!value='$GFX->{$g}[7]' selected>!i;

        my ($sel_l, $sel_f, $sel_g);

            $sel_l = 'selected' if $GFX->{$g}[2] == 3;
            $sel_g = 'selected' if $GFX->{$g}[2] == 1;
            $sel_f = 'selected' if $GFX->{$g}[2] == 2;

        $html .= qq~<tr>
                      <td bgcolor='#EEEEEE' colspan='7'><span style='font-size:12px'>Properties for <font color='red'>$GFX->{$g}[0]</font></span></td>
                    </tr>
                    <tr>
                      <td bgcolor='#FFFFFF'><input type='text' name='$g-LINK' value='$GFX->{$g}[1]' style='width:95%'></td>
                      <td bgcolor='#FFFFFF'><select name='$g-TYPE' style='width:95%'><option value='1' $sel_g>Graphic</option><option value='3' $sel_l>Text Link</option><option value='2' $sel_f>Flash Movie</option></select></td>
                      <td bgcolor='#FFFFFF'><input style='width:95%' type='text' name='$g-WID' value='$GFX->{$g}[4]'></td>
                      <td bgcolor='#FFFFFF'><input style='width:95%' type='text' name='$g-HIE' value='$GFX->{$g}[5]'></td>
                      <td bgcolor='#FFFFFF'><input style='width:95%' type='text' name='$g-ALT' value='$GFX->{$g}[6]'></td>
                      <td bgcolor='#FFFFFF'><input style='width:95%' type='text' name='$g-BORDER' value='$GFX->{$g}[3]'><input type='hidden' name='$g-RN' value='$GFX->{$g}[0]'</td>
                      <td bgcolor='#FFFFFF'><select name='$g-AGN' style='width:95%'>$the_options</select></td>
                    </tr>
                   ~;

    }

    $html .= qq~</table></td></tr>~;

    $html .= $SKIN->td_submit(   NAME => '', VALUE => 'Apply this Edit' );


    $html .= $SKIN->form_end();
    $html .= $SKIN->end_table();

    #+-----------------------------------

    $ADMIN->Output( WHERE => 'STYLES', NAV_ONE => 'Graphics Properties', PRINT => $html);

}

sub do_GFX {
    my ($obj, $db) = @_;

    $ADMIN->Error( DB=>,"", STD=>"", MSG => "Not a valid skin package") unless $iB::IN{'SKIN'};

    require $iB::IN{'SKIN'} . '/Styles.pm' or die $!;
    require $iB::IN{'SKIN'} . '/gfx_data.cfg' or die $!;

    my $GFX = gfx_data->new();
    my $SKN = Styles->new();

    my @incoming = grep { /-LINK$/ } $iB::CGI->param;

    my $n_GFX = {};
    my $n_SKN = {};

    for my $in (@incoming) {
        my $key = $in;
        # Get the form field name
        $key =~ s!-LINK$!!;
        # Get the text/gfx link attr
        my $gfx_resource = $iB::CGI->param($key.'-LINK');
        # Get the TYPE attr
        my $gfx_type     = $iB::CGI->param($key.'-TYPE');
        # Get the WIDTH attr
        my $gfx_width     = $iB::CGI->param($key.'-WID');
        # Get the HEIGHT attr
        my $gfx_height    = $iB::CGI->param($key.'-HIE');
        # Get the BORDER attr
        my $gfx_border    = $iB::CGI->param($key.'-BORDER');
        # Get the NAME attr
        my $gfx_name      = $iB::CGI->param($key.'-RN');
        # Get the ALT attr
        my $gfx_alt       = $iB::CGI->param($key.'-ALT');
        # Get the ALIGN attr
        my $gfx_align     = $iB::CGI->param($key.'-AGN');

        # Start to munge the data

        # Is it a flash movie?
        if ($gfx_type == 2) {
            # Append to the skin file
            my $real_location = '$iB::INFO->{\'IMAGES_URL\'}/Skin/'.$iB::IN{'SKIN'}.'/images/'.$gfx_resource;
            $n_SKN->{$key} = qq~<OBJECT CLASSID="clsid:D27CDB6E-AE6D-11cf-96B8-444553540000" WIDTH=$gfx_width HEIGHT=$gfx_height><PARAM NAME=MOVIE VALUE="$real_location"><PARAM NAME=PLAY VALUE=TRUE><PARAM NAME=LOOP VALUE=TRUE><PARAM NAME=QUALITY VALUE=HIGH><EMBED SRC="$real_location" WIDTH=$gfx_width HEIGHT=$gfx_height PLAY=TRUE LOOP=TRUE QUALITY=HIGH></EMBED></OBJECT>~;
        } elsif
           ($gfx_type == 1) {
            # Is it a graphic?
            my $r_width  = $gfx_width   ne ''  ? ' width="'.$gfx_width.'" '    : '';
            my $r_height = $gfx_height  ne ''  ? ' height="'.$gfx_height.'" '  : '';
            my $r_align  = $gfx_align   ne ''  ? ' align="'.$gfx_align.'" '    : '';
            my $r_border = $gfx_border  ne ''  ? ' border="'.$gfx_border.'" '  : ' border="0" ';
            $gfx_alt =~ s!"!\\"!g;
            $gfx_alt =~ s!'!\\'!g;
            my $r_alt    = ' alt="'.$gfx_alt.'"';
            # Append to the skin file
            my $real_location = '$iB::INFO->{\'IMAGES_URL\'}/Skin/'.$iB::IN{'SKIN'}.'/images/'.$gfx_resource;
            $n_SKN->{$key} = qq~<img src="$real_location"$r_width$r_height$r_border$r_align$r_alt>~;
        } else {
            # It must be a text link...
            $n_SKN->{$key} = $gfx_resource;
        }

        # Append to the data file

        $gfx_alt =~ s!"!\\"!g;

        $n_GFX->{$key} = [ $gfx_name, $gfx_resource, $gfx_type, $gfx_border, $gfx_width, $gfx_height, $gfx_alt, $gfx_align ];

    }


    # Append the master data
    for my $k (keys %{$n_SKN}) {
        $SKN->{$k} = $n_SKN->{$k};
    }

    for my $h (keys %{$n_GFX}) {
        $GFX->{$h} = $n_GFX->{$h};
    }
    
    # Rebuild the modules

    $ADMIN->make_module( FILE        => 'Styles.pm',
                         PKG_NAME    => 'Styles',
                         VALUES      => $SKN,
                         PATH        => $iB::INFO->{'IKON_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/',
                         INTERPOLATE => "yes",
                       );

    $ADMIN->make_module( FILE     => 'gfx_data.cfg',
                         PKG_NAME => 'gfx_data',
                         VALUES   => $GFX,
                         PATH     => $iB::INFO->{'IKON_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/'
                       );

    $ADMIN->write_log( TITLE => 'Graphic/Icon properties edited');

    $ADMIN->static_screen( URL   => "act=styles&CODE=edit&SECTION=$iB::IN{'SECTION'}&SKIN=$iB::IN{'SKIN'}",
                           TITLE => "Properties edited",
                           TEXT  => "The changes were successful"
                         );

}

#########################################################################################################
#
# HTML Routines
#
#########################################################################################################

sub HTML {
    my ($obj, $db) = @_;

    # Load up the skin list config file
    require "SkinList.cfg";
    # Assign the contstructor
    my $list = SkinList->new();

    my $html = $SKIN->title( TITLE => 'Selecting a Package to Edit', TEXT => "" );
    $html .= $SKIN->begin_table();

    $html .= $SKIN->section_header( TITLE => "Available Skin Packages", TEXT => "Simply click on the title of the package you wish to edit");
    $html .= "<tr class='t'><td colspan='2' bgcolor='#FFFFFF'>";

    for my $element (keys %{$list}) {
        
        $html .= qq~
                   <a href='$iB::INFO->{'BOARD_URL'}/ikonboard.$iB::INFO->{'CGI_EXT'}?s=$iB::SESSION&AD=1&act=styles&CODE=do_skin_edit&SECTION=$iB::IN{'SECTION'}&SKIN=$iB::IN{'SKIN'}&PKG=$element' target='BODY' style='font-size:12px'>$list->{$element}[0]</a>
                   <br>$list->{$element}[1]<br><br>
                  ~;
    }

    $html .= "</td></tr>";


    $html .= $SKIN->end_table();

    #+-----------------------------------

    $ADMIN->Output( WHERE => 'STYLES', NAV_ONE => 'HTML Properties', PRINT => $html);

}

#########################################################
#
# Grab all the form data, and print to the module and config files.
#
#########################################################

sub do_HTML {
    my ($obj, $db) = @_;

    $ADMIN->Error( DB=>,"", STD=>"", MSG => "Not a valid skin package") unless $iB::IN{'SKIN'};
    $ADMIN->Error( DB=>,"", STD=>"", MSG => "Not a valid skin section") unless $iB::IN{'SECTION'};
    $ADMIN->Error( DB=>,"", STD=>"", MSG => "Not a valid skin element package") unless $iB::IN{'PKG'};

    # A Mecham Sanity-Saver (TM)

    my $config = $iB::INFO->{'IKON_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/'.$iB::IN{'PKG'}.'.cfg';
    my $module = $iB::INFO->{'IKON_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/'.$iB::IN{'PKG'}.'.pm';

    # Lets make sure they are there, and writable with our user...
    unless (-e $config) {
        $ADMIN->Error( DB=>,"", STD=>"", MSG => "Cannot find $config");
    }
    unless (-w $config) {
        $ADMIN->Error( DB=>,"", STD=>"", MSG => "Cannot write to $config, please CHMOD to 0777");
    }
    unless (-e $module) {
        $ADMIN->Error( DB=>,"", STD=>"", MSG => "Cannot find $module");
    }
    unless (-w $module) {
        $ADMIN->Error( DB=>,"", STD=>"", MSG => "Cannot write to $module, please CHMOD to 0777");
    }

    # Lets kick start the strings that hold the module/config info.

    # Gimmie back my HTML!..
    my $header = $iB::CGI->param('HEADER');
    $header =~ s!&#60;!<!g;
    $header =~ s!&#62;!>!g;

    my $config_info = qq~[=NAME]\n$iB::IN{'PKG'}\n[=HEADER]\n$header\n~;
    my $module_info = qq~package $iB::IN{'PKG'};\n\n$header\n\n~;

    # Grab in the form data (and there is a lot!)
    
    my @input = grep { /^SUB:/ } $iB::CGI->param();

    # Ok, start the loop and process the data.
    # Sitting comfortably?

    for my $i (@input) {
        # Grab the sub routine name..
        my $name = $i;
        $name    =~ s!^SUB:(.+?)$!$1!;
        # Get the subroutine data..
        my $this_sub  = $iB::CGI->param('SUB:'.$name);
        # Get the top info
        my $this_top  = $iB::CGI->param('TOP:'.$name);
        # Get the description info
        my $this_desc = $iB::CGI->param('DESC:'.$name);

        # Sort out the tags...
        $this_sub =~ s!&#60;!<!g;
        $this_sub =~ s!&#62;!>!g;
        # Make tidle's safe
        $this_sub =~ s!~!&#152;!g;
        # Convert $SKIN tags back..
        $this_sub =~ s!<%SKIN:(\w+)%>!\$iB::SKIN->\{'$1'\}!ig;
        # Convert language tags back..
        $this_sub =~ s!<%LANG:(\w+):(\w+)%>!\$$1::lang->\{'$2'\}!ig;
        # Convert $iB::INFO tags back..
        $this_sub =~ s!<%VAR:(\w+)%>!\$iB::INFO->\{'$1'\}!ig;
        # Convert $iB::IN tags back..
        $this_sub =~ s!<%IN:(\w+)%>!\$iB::IN\{'$1'\}!ig;
        # Convert $iB::MEMBER tags back..
        $this_sub =~ s!<%MEMBER:(\w+)%>!\$iB::MEMBER->\{'$1'\}!ig;

        # Remove carriage returns..
        $this_sub =~ s!\r!\n!g;
        # Remove double spaces..
        $this_sub =~ s!\n\n!\n!g;

        # Append to the config file.
        $config_info .= qq~[=SUB-$name]\n#=DESC\n$this_desc\n#=TOP_LINE\n$this_top\n#=BODY\n$this_sub\n~;
        # Append to our module.
        $module_info .= qq~sub $name {\n\t$this_top\nreturn qq\~\n$this_sub\n\~;\n}\n\n~;
    }

    # Dont forget the true value for our require file..

    $module_info .= "\n\n1;";

    # Before we go any further, lets check to make sure we've got a 
    # valid perl module.

    # Save it to a test file..

    open TESTFILE, ">$module.txt" or $ADMIN->Error( DB=>,"", STD=>"", MSG => "Bleh, couldn't save to the test module file ($!)");
    if ($iB::INFO->{'FLOCK'}) {
        flock (TESTFILE, 2);
    }
    print TESTFILE $module_info;
    close TESTFILE;

    {
        # Turn off our die catcher...
        local $SIG{__DIE__} = undef;
        # Try and "require" it
        eval { require "$module.txt"; };
        
        if ($@) {
            unlink "$module.txt";
            my $error = $@;
            $error =~ s!$module.txt!!g;
            $ADMIN->Error( DB=>,"", STD=>"", MSG => "Error: A mistake has been made when editing, this is no longer a valid perl module. No changes have been made to the existing skin modules or configuration files. Check to make sure you haven't distrubed the &lt;%TAG:tags%&gt; or the \$data->{'DATA'} tags <br><br>(Perl says: <span style='color:red'>$error</span>)");
        }
    }

    # All is good then..
    unlink "$module.txt";

    # Lets print out the files..

    open CONFIGFILE, ">$config" or $ADMIN->Error( DB=>,"", STD=>"", MSG => "Bleh, couldn't save to the config file ($!)");
    if ($iB::INFO->{'FLOCK'}) {
        flock (CONFIGFILE, 2);
    }
    print CONFIGFILE $config_info;
    close CONFIGFILE;

    chmod (0777, $config);

    open MODULEFILE, ">$module" or $ADMIN->Error( DB=>,"", STD=>"", MSG => "Bleh, couldn't save to the module file ($!)");
    if ($iB::INFO->{'FLOCK'}) {
        flock (MODULEFILE, 2);
    }
    print MODULEFILE $module_info;
    close MODULEFILE;

    chmod (0777, $module);

    # Yippee done..

    $ADMIN->static_screen( URL   => "act=styles&CODE=edit&SECTION=$iB::IN{'SECTION'}&SKIN=$iB::IN{'SKIN'}&PKG=$iB::IN{'PKG'}",
                           TITLE => "HTML Properties edited",
                           TEXT  => "The changes were successful"
                         );

}


sub do_skin_edit {
    my ($obj, $db) = @_;
    $ADMIN->Error( DB=>,"", STD=>"", MSG => "Not a valid skin package") unless $iB::IN{'SKIN'};
    $ADMIN->Error( DB=>,"", STD=>"", MSG => "Not a valid skin section") unless $iB::IN{'SECTION'};
    $ADMIN->Error( DB=>,"", STD=>"", MSG => "Not a valid skin element package") unless $iB::IN{'PKG'};

    my ($fdata, @config);

    #Lets open the config file...
    my $file = $iB::INFO->{'IKON_DIR'}.'Skin/'.$iB::IN{'SKIN'}.'/'.$iB::IN{'PKG'}.'.cfg';
    {
      local $/ = undef;
      open FILE, $file or die "Cannot load file: $file ($!)";
      $fdata = <FILE>;
      close FILE;
    }
    # Make sure \r's are \n's
    $fdata =~ s!\r!\n!g;
    
    @config = split /\n/, $fdata;

    my ($header, $name, $set_state, $flag);
    my $subs = {};
    #Lets sort this config file out...
    # (When Stewart Campbell saw that comment above
    # he said "It sounds like you're about to punch it)
    # I'm tempted...
    for my $l (split /\n/, $fdata) {
        # Preserve newlines
        $l .= "\n";
        # Do we have a section header?
        if ($l =~ /^\[=(.+?)\]$/) {
            # Set the global loop handler
            # so on the next loop we know what to expect
            $set_state = $1;
            # Reset the flag
            $flag = 0;
            # Next line please!
            next;
        }
        if ($set_state) {
            # Is it a header?
            if ($set_state eq 'HEADER') {
                $header .= $l;
                next;
            }
            # Are we pulling in a subroutine?
            if ($set_state =~ /SUB-(\S+)$/) {
                if ($flag == 1) {
                    $subs->{$1} .= $l;
                    next;
                } else {
                    #Initialsie the sub hash ref.
                    $subs->{$1} .= $l;
                    # Set the flag so we know we are pulling
                    # in a subroutine
                    $flag = 1;
                    next;
                }
            }
        }
    }



    my $html .= $SKIN->title( TITLE => 'Editing HTML Properties', TEXT => "You may edit the HTML in this section. You may use HTML.");
    $html .= $SKIN->begin_table();
    $html .= $SKIN->form_start();
    $html .= $SKIN->hidden_fields( {    act     => 'styles',
                                        CODE    => 'doedit',
                                        SECTION => 'HTML',
                                        SKIN    => $iB::IN{'SKIN'},
                                        PKG     => $iB::IN{'PKG'},
                                      } );

    $html .= $SKIN->section_header( TITLE => "Module Header Code", TEXT => "Leave this alone unless you know what you're doing");

    # Make HTML safe..
    $header =~ s!<!&#60;!g;
    $header =~ s!>!&#62;!g;

    $html .= $SKIN->td_textarea( TEXT  => "<span style='color:red'>Module Header Information</span>",
                                 NAME  => 'HEADER',
                                 VALUE => $header,
                                 REG   => 1,
                                 NO_PARSE => 1,
                                 ROWS  => 5,
                               );

    # Initialise global counter
    my $cnt = 0;

    for my $s (keys %{$subs}) {

        # Split the text up some more...
        my (%section, $top, $desc, $the_sub);
        # Suck it up into an array
        my @complete = split /\n/, $subs->{ $s };
        # Start the loop.
        for my $i (@complete) {
            if ($i =~ /^#=TOP_LINE/) {
                # Apparently so, lets set the flag
                $top = 1;
                #next line please...
                next;
            } elsif ($i =~ /^#=DESC/) {
                $desc = 1;
                $top  = 0;
                next;
            } elsif ($i =~ /^#=BODY/) {
                $desc    = 0;
                $top     = 0;
                $the_sub = 1;
                next;
            }
            # Are we looking for a top line?
            if ($top == 1) {
                $section{'top_line'} .= $i;
                # any more?
                next;
            } elsif ($desc == 1) {
                # That means we must have all the top line info..
                $top = 0;
                $section{'description'} = $1;
                # Descriptions are on one line only, so..
                $desc = 0;
                next;
            } elsif ($the_sub == 1) {
                # We must have the top line info and desc sucked up then..
                $desc = 0;
                $top  = 0;
                # Add the line to the sub...
                $section{'sub'} .= "\n".$i;
                next;
            }
        }


        $html .= $SKIN->section_header( TITLE => "Editable code for <span style='color:red'>$s</span>", TEXT => $section{'description'} );
        $html .= "<tr><td colspan='2'><input type='hidden' name='DESC:$s' value='$section{'description'}'></td></tr>";
    
        $html .= $SKIN->td_textarea( TEXT  => "Subroutine Header",
                                     NAME  => "TOP:$s",
                                     VALUE => $section{'top_line'},
                                     NO_PARSE => 1,
                                     ROWS  => 5
                                    );

        # Convert $SKIN tags into something less ugly..
        $section{'sub'} =~ s!\$iB::SKIN->\{'(\w+)'\}!<%SKIN:$1%>!ig;
        # Convert language tags into something less ugly..
        $section{'sub'} =~ s!\$(\w+)::lang->\{'(\w+)'\}!<%LANG:$1:$2%>!ig;
        # Convert $iB::INFO tags into something less ugly..
        $section{'sub'} =~ s!\$iB::INFO->\{'(\w+)'\}!<%VAR:$1%>!ig;
        # Convert $iB::IN tags into something less ugly..
        $section{'sub'} =~ s!\$iB::IN\{'(\w+)'\}!<%IN:$1%>!ig;
        # Convert $iB::MEMBER tags into something less ugly..
        $section{'sub'} =~ s!\$iB::MEMBER->\{'(\w+)'\}!<%MEMBER:$1%>!ig;

        # Make HTML safe..
        $section{'sub'} =~ s!<!&#60;!g;
        $section{'sub'} =~ s!>!&#62;!g;

        $html .= $SKIN->td_textarea( TEXT  => "The HTML source",
                                     NAME  => "SUB:$s",
                                     VALUE => $section{'sub'},
                                     SPAN => 1,
                                     NO_PARSE => 1,
                                     ROWS  => 20
                                    );


    }

    $html .= $SKIN->td_submit(   NAME => '', VALUE => 'Apply this Edit' );


    $html .= $SKIN->form_end();
    $html .= $SKIN->end_table();

    #+-----------------------------------

    $ADMIN->Output( WHERE => 'STYLES', NAV_ONE => 'HTML Properties', PRINT => $html);

}


#########################################################################################################
#
# Process Jumper
#
#########################################################################################################


sub process {
    my ($obj, $db) = @_;

    my $CodeNo = $iB::IN{'CODE'};

    my %Mode = ( 'edit'          => \&edit,
                 'doedit'        => \&doedit,
                 'do_skin_edit'  => \&do_skin_edit,
               );
    $Mode{$CodeNo} ? $Mode{$CodeNo}->($obj,$db) : splash($obj,$db);
} 




1;