package Admin::Convert_ib;
use strict;
#################################################################################
# Ikonboard v3 by Jarvis Entertainment Group, Inc.
#
# No part of this script can be used outside Ikonboard without prior consent.
#
# More information available from <ib-license@jarvisgroup.net>
# (c)2001 Jarvis Entertainment Group, Inc.
# 
# http://www.ikonboard.com | http://www.jarvisgroup.net
#
# Please read the license for more information
#
# 
# Script Author: Matthew Mecham <matt@ikonboard.com>
#
#################################################################################

# BENDER: "Hey rich girl, look at my head! It's okay to look at my head, I've got
#          a big 'ol head and then, hey! ho!...... OK, shows over, I'm tired."

BEGIN {
    require 'Lib/FUNC.pm';
    require 'Lib/ADMIN.pm';
    require 'Admin/SKIN.pm';
    require 'Boardinfo.pm' or die "Cannot load Module: $!";
    require iTextparser;
    @AnyDBM_File::ISA = qw(DB_File GDBM_File NDBM_File SDBM_File);
}

use AnyDBM_File;
use Fcntl qw(:DEFAULT :flock);

my $SKIN  = Admin::SKIN->new();
my $std   = FUNC::STD->new();
my $ADMIN = FUNC::ADMIN->new();
my $INFO  = Boardinfo->new();
my $mem   = FUNC::Member->new();
my $txt   = iTextparser->new();

sub new {
    my $pkg = shift;
    my $obj = {};
    bless $obj, $pkg;
    return $obj;
}

sub splash {
    my ($obj, $db) = @_;

    # Have we done this before?
    my $saved = $obj->_load_config;

    my $mem_groups = $db->query( TABLE      => 'mem_groups',
                                 COLUMNS    => ['ID', 'TITLE', 'CAN_REMOVE'],
                                 SORT_KEY   => 'TITLE',
                                 SORT_BY    => 'A-Z',
                               );

    my @groups;

    for (@{$mem_groups}) {
        push @groups, { NAME => $_->{'TITLE'}, VALUE => $_->{'ID'} };
    }


    use Cwd;
    my $pth     = $saved->{'LOCATION'} || cwd();

    my $posts   = $saved->{'POST_COUNT'} || '200';
    my $members = $saved->{'MEM_COUNT'}  || '200';

    my $html  = $SKIN->title( TITLE => 'Converting Ikonboard 2.1.9 to Ikonboard 3', TEXT => "This convertor will attempt to convert all your Ikonboard 2 data to the new format.<br>Please read through the instructions very carefully" );
       $html .= $SKIN->begin_table();
       $html .= $SKIN->form_start();
       $html .= $SKIN->hidden_fields( { act   => 'convert',
                                        CODE  => 'one'
                                      } );

    #+-----------------------------------

    $html .= $SKIN->section_header( TITLE => "Set up the convertor" );

    $html .= $SKIN->td_input ( TEXT => 'Where is your ikonboard 2?<br>&nbsp;&nbsp;&nbsp;&nbsp;(Enter the full path to where ikon.lib resides)', NAME => 'LOCATION',  VALUE=> $pth, REQ => 1);
    $html .= $SKIN->td_input ( TEXT => 'Number of members to convert at a time', NAME => 'MEM_COUNT',   VALUE=> $members, REQ => 1);
    $html .= $SKIN->td_input ( TEXT => 'Number of topics to convert at a time'  , NAME => 'POST_COUNT',  VALUE=> $posts  , REQ => 1);

    $html .= $SKIN->td_select( TEXT     => 'If an old iB2 member name matches an iB3 membername..',
                               NAME     => 'OVERWRITE',
                               SIZE     => '',
                               REQ      => 1,
                               VALUES   => $saved->{'OVERWRITE'},
                               DATA     => [ { VALUE => 1, NAME => 'Overwrite the current iB3 member'  },
                                             { VALUE => 0, NAME => 'Keep the current iB3 member' },
                                           ]
                             );


    $html  .= $SKIN->section_header(   TITLE => 'User Default Grouping',
                                       TEXT  => "Please choose which group to put the following members into",
                           );

    $html .= $SKIN->td_select( TEXT     => "Normal members",
                               NAME     => 'MEMBER_GROUP',
                               REQ      => 1,
                               VALUES   => $saved->{'MEMBER_GROUP'} || $iB::INFO->{'MEMBER_GROUP'},
                               DATA     => \@groups,
                             );

    $html .= $SKIN->td_select( TEXT     => "Administrators",
                               NAME     => 'ADMIN_GROUP',
                               REQ      => 1,
                               VALUES   => $saved->{'ADMIN_GROUP'} || $iB::INFO->{'SUPAD_GROUP'},
                               DATA     => \@groups,
                             );

    $html .= $SKIN->td_select( TEXT     => "Moderators",
                               NAME     => 'MOD_GROUP',
                               REQ      => 1,
                               VALUES   => $saved->{'MOD_GROUP'} || $iB::INFO->{'MEMBER_GROUP'},
                               DATA     => \@groups,
                             );

    $html .= $SKIN->td_submit(   NAME => '', VALUE => 'Proceed to the next step' );


    $html .= $SKIN->form_end();
    $html .= $SKIN->end_table();

    #+-----------------------------------

    $ADMIN->Output( WHERE => 'DATABASE', NAV_ONE => 'Converting from an older Ikonboard', PRINT => $html);

}


###################################################################
#
# STEP ONE => Check and write details, convert members if needed
#
###################################################################


sub step_one {
    my ($obj, $db) = @_;

    unless ($iB::IN{'skip'}) {

        # Remove trailing slash from LOCATION
        $iB::IN{'LOCATION'} =~ s!(?:[/\\]+)$!!;
    
        # Check for blanks
        $ADMIN->Error( DB => $db, STD => $std, MSG => "You must enter a path for the old ikonboard location")         unless $iB::IN{'LOCATION'};
        $ADMIN->Error( DB => $db, STD => $std, MSG => "You must enter a number for the members to convert at a time") unless $iB::IN{'MEM_COUNT'};
        $ADMIN->Error( DB => $db, STD => $std, MSG => "You must enter a number for the posts to convert at a time")   unless $iB::IN{'POST_COUNT'};
    
        # Check for old ikonboard
        my $data_found  = (-e $iB::IN{'LOCATION'}.'/data/allforums.cgi') ? 1 : 0;
        my $board_found = (-e $iB::IN{'LOCATION'}.'/ikon.lib')           ? 1 : 0;
    
        unless ($data_found || $board_found) {
            $ADMIN->Error( DB => $db, STD => $std, MSG => "Cannot find the ikonboard v2.1.9 installation!<br>Please check that the paths above are correct and that 'ikon.lib' is still present in the old installation");
        }

        # Write to the config file
        my $config = {};
    
        for (qw[LOCATION MEM_COUNT POST_COUNT OVERWRITE MEMBER_GROUP ADMIN_GROUP MOD_GROUP]) {
            $config->{$_} = $iB::IN{$_};
        }
    
        $obj->_write_config($config);
    }

    my $mem_done = $obj->_check_convert;

    my $html  = $SKIN->title( TITLE => 'Converting Ikonboard 2.1.9 to Ikonboard 3', TEXT => "This convertor will attempt to convert all your Ikonboard 2 data to the new format.<br>Please read through the instructions very carefully" );
       $html .= $SKIN->begin_table();
    $html .= "<tr class='t'><td colspan='2' bgcolor='#FFFFFF'><span style='font-size:12px'><b>Ikonboard 2.1.9 installation found</b><br>";

    if ($mem_done->{'MEMBERS'}) {
        my $cats   = $mem_done->{'CATS'}   ? '<span style=\'color:red\'>[ Category Conversion Successful ]</span>'               : '';
        my $mods   = $mem_done->{'MODS'}   ? '<span style=\'color:red\'>[ Moderator Conversion Successful ]</span>Convert More?' : '';
        my $forums = $mem_done->{'FORUMS'} ? '<span style=\'color:red\'>[ Forum Conversion Successful ]</span> Convert More?'    : '';
        my $posts  = $mem_done->{'POSTS'}  ? '<span style=\'color:red\'>[ Post Conversion Successful ]</span> Convert More?'    : '';
        $html .= qq~</span><br><br>
                    &gt; <a href='$iB::INFO->{'BOARD_URL'}/ikonboard.$iB::INFO->{'CGI_EXT'}?s=$iB::SESSION&AD=1&act=convert&CODE=members' target='BODY' style='font-size:12px'>Convert Members</a> <span style='color:red'>[ Member Conversion Successful ]</span><br>This will allow you to import existing members from your Ikonboard 2 installation. The convertor will start after you click the above link, so go back and check your convertor set up preferences if you're unsure.
                    <br><br>&gt; <a href='$iB::INFO->{'BOARD_URL'}/ikonboard.$iB::INFO->{'CGI_EXT'}?s=$iB::SESSION&AD=1&act=convert&CODE=cats' target='BODY' style='font-size:12px'>Convert Category Information</a>$cats<br>This will allow you to create new categories based on the ikonboard 2 you currently have. This will NOT overwrite any current categories, it will create new categories in addition to the ones you currently have. Clicking on this link will take you to an options screen.
                    <br><br>&gt; <a href='$iB::INFO->{'BOARD_URL'}/ikonboard.$iB::INFO->{'CGI_EXT'}?s=$iB::SESSION&AD=1&act=convert&CODE=forums' target='BODY' style='font-size:12px'>Convert Forum Information</a>$forums<br>This will allow you to create new forums based on the ikonboard 2 you currently have. This will NOT overwrite any current forums, it will create new forums in addition to the ones you currently have. Clicking on this link will take you to an options screen. It is recommended that you convert and set-up any categories you wish to convert first before converting forums.
                    <br><br>&gt; <a href='$iB::INFO->{'BOARD_URL'}/ikonboard.$iB::INFO->{'CGI_EXT'}?s=$iB::SESSION&AD=1&act=convert&CODE=posts' target='BODY' style='font-size:12px'>Convert Topics and Posts</a>$posts<br>This will allow you to import posts and topic data from your Ikonboard 2 installation. Clicking on this link will take you to an options screen.
                    <br><br>&gt; <a href='$iB::INFO->{'BOARD_URL'}/ikonboard.$iB::INFO->{'CGI_EXT'}?s=$iB::SESSION&AD=1&act=convert&CODE=mods' target='BODY' style='font-size:12px'>Convert Moderator Information</a>$mods<br>This will allow you to convert across your current moderator information. Clicking on this link will take you to an options screen.
                   ~;
    } else {
        $html .= qq~You must convert members before converting posts or forums</span><br><br>
                    &gt; <a href='$iB::INFO->{'BOARD_URL'}/ikonboard.$iB::INFO->{'CGI_EXT'}?s=$iB::SESSION&AD=1&act=convert&CODE=members' target='BODY' style='font-size:12px'>Convert Members</a><br>This will allow you to import existing members from your Ikonboard 2 installation. The convertor will start after you click the above link, so go back and check your convertor set up preferences if you're unsure.
                  ~;
                
    }

    $html .= $SKIN->end_table();

    $ADMIN->Output( WHERE => 'DATABASE', NAV_ONE => 'Converting from an older Ikonboard', PRINT => $html);

}


###################################################################
#
# MEMBERS => Convert the members
#
###################################################################

sub members {
    my ($obj, $db) = shift;

    # Get the config file
    my $saved = $obj->_load_config;

    $ADMIN->Error( DB => $db, STD => $std, MSG => "Could not find the ib2 converter datafile")   unless $saved->{'LOCATION'};

    # Still here? Ok, lets get the member names.

    opendir DIR, $saved->{'LOCATION'}.'/members' or $ADMIN->Error( DB => $db, STD => $std, MSG => "Could not open the iB2 members directory");
    # Be a lazy assed coder and suck all the members into an array
    # I remember when I first saw the code for reading a directory
    # I seriously thought that it actually read in all the file data
    # with it. I was paranoid about thrashing the server by reading
    # a complete directory at once.
    #
    # Once the prozac kicked in, I realised how stupid I was

    my @members =   sort { $a cmp $b }
                    grep { /\.cgi$/  } readdir DIR;
    closedir DIR;

    my $m_total = scalar(@members);

    # Make sure we have a valid start variable
    $iB::IN{'st'} ||= 0;
    # Get the end variable
    my $end = $iB::IN{'st'} + $saved->{'MEM_COUNT'};

    $end = $m_total if $end > $m_total;

    # Get the array slice we need
    @members = @members[$iB::IN{'st'} .. $end];
    # Increment the $end variable to use later as our start variable
    ++$end;

    my $base_dir = $INFO->{'DB_DIR'}.'member_profiles';

    # =tie a hash to the names index DB
    tie (my %N_INDEX, $AnyDBM_File::ISA[0], "$base_dir/MEMBER_NAME.idx", O_RDWR|O_CREAT, 0777);
    # =tie a hash to the email index DB
    tie (my %E_INDEX, $AnyDBM_File::ISA[0], "$base_dir/MEMBER_EMAIL.idx", O_RDWR|O_CREAT, 0777);
    # =tie a hash to the members DB
    tie (my %MEMS,  $AnyDBM_File::ISA[0], "$base_dir/member_profiles.db", O_RDWR|O_CREAT, 0777);
    # =tie a puppy to a lamp post
    #tie (my %PUPPY, 'to a lamp post', "Corner of 5th and Main Street", O_NO|ESCAPE, 08:00);
    # Just kidding. Tying up puppies is a bad thing boys and girls!
    # =Set up a generic ID
    my $id = $iB::IN{'id'} || time;
    # =Load the config file
    $obj->_load_db_cfg('member_profiles');
    # =Open the filehandle for the installer file
    open INSTALL, ">>".$INFO->{'DB_DIR'}.'Temp/mem_install.txt';
    print INSTALL "Starting Member conversion....\n------------------------\n\n";

    # Here we go loopy, loo!

    for my $m (@members) {
        # Open, suck, split and return the data
        # as hash ref
        my $r_mem = $obj->_split_mem($m, $saved);
        print INSTALL "Attempting to process $r_mem->{'MEMBER_NAME'}...\n";
        next unless $r_mem->{'MEMBER_NAME'};
        # Check to see if a member name exists, and if so, what to do.
        unless ($saved->{'OVERWRITE'}) {
            if (exists $N_INDEX{ $r_mem->{'MEMBER_NAME'} }) {
                print INSTALL "Skipping $r_mem->{'MEMBER_NAME'} as it already exists in the iB3 Database...\n";
                next;
            }
        }
        # Encrypt the password
        $r_mem->{'MEMBER_PASSWORD'} = $mem->Crypt($r_mem->{'MEMBER_NAME'},$r_mem->{'MEMBER_PASSWORD'});
        # Sort out the member ID
        ++$id;
        $r_mem->{'MEMBER_ID'} = ord( substr($r_mem->{'MEMBER_NAME'},0,1) ).'-'.$id;
        # Insert the members profile...
        $MEMS{$r_mem->{'MEMBER_ID'}}       = $obj->_encode_record($r_mem);
        # Insert into names index
        $N_INDEX{$r_mem->{'MEMBER_NAME'}}  = $r_mem->{'MEMBER_ID'};
        # Insert into email index
        $E_INDEX{$r_mem->{'MEMBER_EMAIL'}} = $r_mem->{'MEMBER_ID'};
        # Print the log file entry
        print INSTALL "$r_mem->{'MEMBER_NAME'} added successfully with an ID of $r_mem->{'MEMBER_ID'} ...\n";
        # That's all folks!
    }

    # close the log filehandle...
    close INSTALL;
    # close the DBs
    untie (%MEMS);
    untie (%N_INDEX);
    untie (%E_INDEX);

    # What to do next..

    # If we've processed the last of the members, go to the "done" screen
    if ($end >= $m_total) {
       open LOCKFILE, ">".$INFO->{'DB_DIR'}.'Temp/members.lockfile';
       print LOCKFILE "Done: ".time."\n";
       close LOCKFILE;
       $ADMIN->redirect( URL => "act=convert&CODE=one&skip=1", TEXT => 'Finishing up the member convertion process');
    } else {
        $id += 10; # Just to be safe...
       $ADMIN->redirect( URL => "act=convert&CODE=members&st=$end&id=$id", TEXT => 'Processing the next batch...');
    }
}

###################################################################
#
# CATEGORY set-up
#
###################################################################


sub cats {
    my ($obj, $db) = @_;

    # Get the config file
    my $saved = $obj->_load_config;

    $ADMIN->Error( DB => $db, STD => $std, MSG => "Could not find the ib2 converter datafile")   unless $saved->{'LOCATION'};

    # Get iB2 cats as an array reference holding associative array references

    my $old_cats = $obj->_get_old_cats($saved);

    my $html  = $SKIN->title( TITLE => 'Converting Ikonboard 2 to Ikonboard 3', TEXT => "Converting Categories" );
       $html .= $SKIN->begin_table();
       $html .= $SKIN->form_start();
       $html .= $SKIN->hidden_fields( { act   => 'convert',
                                        CODE  => 'd_cats'
                                      } );

    $html .= $SKIN->section_header( TITLE => "Please check the box for each category you'd like to import", TEXT => "NOTE: This will ONLY import the category information, NOT the forums and posts" );

    $html .= qq~
         <table cellspacing='1' cellpadding='0' width='100%' align='center' border='0' bgcolor='#333366'>
             <tr>
                 <td>
                 <table cellspacing='1' cellpadding='2' width='100%' align='center' border='0'>
                 <tr>
                   <td bgcolor='#666699' width='10%' align='center'><font class='t'>&nbsp;</font></td>
                   <td bgcolor='#666699' width='90%'><font class='t'><b>iB 2 Category Name</b></font></td>
                 </tr>
               ~;


    for my $f (@{$old_cats}) {
        next unless defined $f->{'CAT_POS'};
        $html .= qq~
                 <tr>
                   <td bgcolor='#FFFFFF' align='center'><input type='checkbox' name='c' value='$f->{'CAT_POS'}' style='border:none' checked></td>
                   <td bgcolor='#FFFFFF'><font class='t'>$f->{'CAT_NAME'}</font><input type='hidden' name='CN_$f->{'CAT_POS'}' value='$f->{'CAT_NAME'}'></td>
                 </tr>
                 ~;
    }

    $html .= qq~
            <tr>
            <td colspan='6' align='center' bgcolor='#FFFFFF'><br><input type='submit' value='IMPORT SELECTED CATEGORIES'></td>
            </tr>
            </table>
         </td>
        </tr>
      </table>
     </form>
    </td>
    </tr>
    </table>
    ~;
    $ADMIN->Output( WHERE => 'DATABASE', NAV_ONE => 'Converting from an older Ikonboard', PRINT => $html);
}

sub do_cats {
    my ($obj, $db) = @_;

    my $cats;
    my $cat   = $db->query(TABLE     => 'categories' );

    for my $ct (@{$cat}) {
        if ($ct->{'CAT_ID'} > $cats) {
            $cats = $ct->{'CAT_ID'};
        }
    }

    ++$cats;

    my @cat_ids =  $iB::CGI->param('c');

    for my $c (@cat_ids) {

        $db->insert( TABLE  => 'categories',
                     VALUES => { 
                                 CAT_POS     => $cats,
                                 CAT_STATE   => 1,
                                 CAT_NAME    => $iB::IN{'CN_'.$c},
                                 CAT_DESC    => '',
                                 SUB_CAT_ID  => '-',
                                 VIEW        => '*'
                               }
                  );
        # Increment the CAT_ID
        ++$cats;
    }

       open LOCKFILE, ">".$INFO->{'DB_DIR'}.'Temp/categories.lockfile';
       print LOCKFILE "Done: ".time."\n";
       close LOCKFILE;
       $ADMIN->redirect( URL => "act=convert&CODE=one&skip=1", TEXT => 'Finishing up the category conversion process');
}


###################################################################
#
# FORUM set-up
#
###################################################################


sub forums {
    my ($obj, $db) = @_;

    # Get the config file
    my $saved = $obj->_load_config;

    $ADMIN->Error( DB => $db, STD => $std, MSG => "Could not find the ib2 converter datafile")   unless $saved->{'LOCATION'};

    # Get iB2 forums as an array reference holding associative array references

    my $old_forums = $obj->_get_old_forums($saved);

    # Sort out the cats stuff

    my $cats       = $db->query( TABLE     => 'categories',
                                 COLUMNS   => ['CAT_NAME', 'CAT_ID'],
                                 SORT_KEY  => 'CAT_POS',
                                 SORT_BY   => 'A-Z',
                                 MATCH     => 'ALL',
                               ) || die $db->{'error'};

    my @cat_array;
    my $cat_select;

    for (@{$cats}) {
        $cat_select .= qq!<option value="$_->{'CAT_ID'}">$_->{'CAT_NAME'}</option>!;
    }

    my $html  = $SKIN->title( TITLE => 'Converting Ikonboard 2 to Ikonboard 3', TEXT => "Converting Forum statistics" );
       $html .= $SKIN->begin_table();
       $html .= $SKIN->form_start();
       $html .= $SKIN->hidden_fields( { act   => 'convert',
                                        CODE  => 'd_forums'
                                      } );

    $html .= $SKIN->section_header( TITLE => "Please check the box for each forum you'd like to import and select a category to import into", TEXT => "NOTE: This will ONLY import the forum information, NOT the forums and posts" );

    $html .= qq~
         <table cellspacing='1' cellpadding='0' width='100%' align='center' border='0' bgcolor='#333366'>
             <tr>
                 <td>
                 <table cellspacing='1' cellpadding='2' width='100%' align='center' border='0'>
                 <tr>
                   <td bgcolor='#666699' width='10%' align='center'><font class='t'>&nbsp;</font></td>
                   <td bgcolor='#666699' width='50%'><font class='t'><b>iB 2 Forum Name</b></font></td>
                   <td bgcolor='#666699' width='50%'><font class='t'><b>Import into which category?</b></font></td>
                 </tr>
               ~;


    for my $f (@{$old_forums}) {
        next unless defined $f->{'FORUM_NAME'};
        $f->{'FORUM_DESC'} =~ s!\'!&#39;!g;
        my $temp_c = $cat_select;
        $temp_c =~ s!>$f->{'t_CAT'}</o!selected>$f->{'t_CAT'}</o!g;
        $html .= qq~
                 <tr>
                   <td bgcolor='#FFFFFF' align='center'><input type='checkbox' name='f' value='$f->{'t_ID'}' style='border:none' checked></td>
                   <td bgcolor='#FFFFFF'><font class='t'>$f->{'FORUM_NAME'}</font>
                       <input type='hidden' name='FN_$f->{'t_ID'}' value='$f->{'FORUM_NAME'}'>
                       <input type='hidden' name='FD_$f->{'t_ID'}' value='$f->{'FORUM_DESC'}'>
                       <input type='hidden' name='FH_$f->{'t_ID'}' value='$f->{'FORUM_HTML'}'>
                       <input type='hidden' name='FC_$f->{'t_ID'}' value='$f->{'FORUM_IBC'}'>
                   </td>
                   <td bgcolor='#FFFFFF'><select name='FS_$f->{'t_ID'}'>$temp_c</select></td>
                 </tr>
                 ~;
    }

    $html .= qq~
            <tr>
            <td colspan='6' align='center' bgcolor='#FFFFFF'><br><input type='submit' value='IMPORT SELECTED FORUMS'></td>
            </tr>
            </table>
         </td>
        </tr>
      </table>
     </form>
    </td>
    </tr>
    </table>
    ~;
    $ADMIN->Output( WHERE => 'DATABASE', NAV_ONE => qq!<a href='$iB::INFO->{'BOARD_URL'}/ikonboard.$iB::INFO->{'CGI_EXT'}?s=$iB::SESSION&AD=1&act=convert' target='BODY'>Convertor options</a>!, NAV_TWO => 'Converting from an older Ikonboard', PRINT => $html);
}


sub d_forums {
    my ($obj, $db) = @_;

    $ADMIN->Error( DB => $db, STD => $std, MSG => "No forums to convert!")   unless $iB::CGI->param('f');

    my $forums = $db->query( TABLE  => 'forum_info');

    my $forum = 0;

    for my $f (@{$forums}) {
        if ($f->{'FORUM_ID'} > $forum) {
            $forum = $f->{'FORUM_ID'};
        }
    }
    
    ++$forum;

    my @r_forums = $iB::CGI->param('f');

    for my $rf (@r_forums) {


        $db->create_table( TABLE => 'forum_posts',
                           DBID  => 'f'.$forum,
                         );
    
    
        $db->create_table( TABLE => 'forum_subscriptions',
                           DBID  => 'f'.$forum,
                         );                  
    
    
        $iB::IN{'FORUM_NAME'} = $SKIN->htmlalize($iB::IN{'FN_'.$rf});
        $iB::IN{'FORUM_DESC'} = $iB::CGI->param('FD_'.$rf);
        $iB::IN{'FORUM_DESC'} =~ s!&#39;!'!g;
        $iB::IN{'FORUM_DESC'} = $SKIN->htmlalize($iB::IN{'FORUM_DESC'});
      
        $db->insert( TABLE  => 'forum_info', 
                     VALUES => { FORUM_ID                => $forum,
                                 FORUM_TOPICS            => 0,
                                 FORUM_POSTS             => 0,
                                 FORUM_LAST_POST         => undef,
                                 FORUM_LAST_POSTER       => undef,
                                 FORUM_LAST_POSTER_N     => undef,
                                 FORUM_NAME              => $iB::IN{'FORUM_NAME'},
                                 FORUM_DESC              => $iB::IN{'FORUM_DESC'},
                                 FORUM_IBC               => $iB::IN{'FC_'.$rf},
                                 FORUM_POSITION          => $forums,
                                 FORUM_HTML              => $iB::IN{'FH_'.$rf},
                                 FORUM_STATUS            => 1,
                                 FORUM_START_THREADS     => '*',
                                 FORUM_REPLY_THREADS     => '*',
                                 FORUM_VIEW_THREADS      => '*',
                                 FORUM_PROTECT           => '',
                                 CATEGORY                => $iB::IN{'FS_'.$rf},
                                 L_TOPIC_TITLE           => undef,
                                 L_TOPIC_ID              => undef,
                                 SORT_KEY                => undef,
                                 SORT_ORDER              => undef,
                                 PRUNE_DAYS              => undef,
                                 ALLOW_ATTACH            => 0,
                                 SHOW_RULES              => 0,
                               }
                  );

        # Increment the global $forum id integar

        ++$forum;
    
        $ADMIN->write_log( TITLE => 'Forum Added', EXTRA => "Forum ID: $forum");

    } # end loop


    $obj->{'TOTAL_CATS'}   = $db->query(TABLE     => 'categories',
                                        SORT_KEY  => 'CAT_POS',
                                        SORT_BY   => 'A-Z',
                                        MATCH     => 'ALL',
                                      ) || die $db->{'error'};

    $obj->{'TOTAL_FORUMS'} = $db->query(TABLE     => 'forum_info',
                                        SORT_KEY  => 'FORUM_POSITION',
                                        SORT_BY   => 'A-Z',
                                        MATCH     => 'ALL'
                                      ) || die $db->{'error'};


    $std->build_forumjump( DB     => $db,
                           CATS   => $obj->{'TOTAL_CATS'},
                           FORUMS => $obj->{'TOTAL_FORUMS'}
                         );

    open LOCKFILE, ">".$INFO->{'DB_DIR'}.'Temp/forums.lockfile';
    print LOCKFILE "Done: ".time."\n";
    close LOCKFILE;
    $ADMIN->redirect( URL => "act=convert&CODE=one&skip=1", TEXT => 'Finishing up the forum conversion process');


}

###################################################################
#
# MODERATOR set-up
#
###################################################################


sub mods {
    my ($obj, $db) = @_;

    # Get the config file
    my $saved = $obj->_load_config;

    $ADMIN->Error( DB => $db, STD => $std, MSG => "Could not find the ib2 converter datafile")   unless $saved->{'LOCATION'};

    # Get iB2 forums as an array reference holding associative array references

    my $old_forums = $obj->_get_old_forums($saved);

    # Sort out the forums stuff

    my $forums   = $db->query( TABLE     => 'forum_info',
                               COLUMNS   => ['FORUM_NAME', 'FORUM_ID'],
                             ) || die $db->{'error'};

    my @cat_array;
    my $forum_select;

    for (@{$forums}) {
        $forum_select .= qq!<option value="$_->{'FORUM_ID'}">$_->{'FORUM_NAME'}</option>!;
    }

    my $html  = $SKIN->title( TITLE => 'Converting Ikonboard 2 to Ikonboard 3', TEXT => "Converting Moderator Information" );
       $html .= $SKIN->begin_table();
       $html .= $SKIN->form_start();
       $html .= $SKIN->hidden_fields( { act   => 'convert',
                                        CODE  => 'd_mods'
                                      } );

    $html .= $SKIN->section_header( TITLE => "Please check the box for each moderator you'd like to import and select an iB3 forum to import into", TEXT => "NOTE: This will only convert the moderators, no other information will be imported" );

    $html .= qq~
         <table cellspacing='1' cellpadding='0' width='100%' align='center' border='0' bgcolor='#333366'>
             <tr>
                 <td>
                 <table cellspacing='1' cellpadding='2' width='100%' align='center' border='0'>
                 <tr>
                   <td bgcolor='#666699' width='10%' align='center'><font class='t'>&nbsp;</font></td>
                   <td bgcolor='#666699' width='30%'><font class='t'><b>iB 2 Forum Name</b></font></td>
                   <td bgcolor='#666699' width='30%'><font class='t'><b>iB 2 Forum Moderators</b></font></td>
                   <td bgcolor='#666699' width='30%'><font class='t'><b>Import into which forum?</b></font></td>
                 </tr>
               ~;


    for my $f (@{$old_forums}) {
        next unless length($f->{'t_MODS'}) > 1;
        my $temp_c = $forum_select;
        $temp_c =~ s!>$f->{'FORUM_NAME'}</o!selected>$f->{'FORUM_NAME'}</o!g;
        $html .= qq~
                 <tr>
                   <td bgcolor='#FFFFFF' align='center'><input type='checkbox' name='f' value='$f->{'t_ID'}' style='border:none' checked></td>
                   <td bgcolor='#FFFFFF'><font class='t'>$f->{'FORUM_NAME'}</font>
                       <input type='hidden' name='FM_$f->{'t_ID'}' value='$f->{'t_MODS'}'>
                   </td>
                   <td bgcolor='#FFFFFF'><font class='t'>$f->{'t_MODS'}</font>
                   <td bgcolor='#FFFFFF'><select name='FS_$f->{'t_ID'}'>$temp_c</select></td>
                 </tr>
                 ~;
    }

    $html .= qq~
            <tr>
            <td colspan='6' align='center' bgcolor='#FFFFFF'><br><input type='submit' value='IMPORT SELECTED MODERATORS'></td>
            </tr>
            </table>
         </td>
        </tr>
      </table>
     </form>
    </td>
    </tr>
    </table>
    ~;
    $ADMIN->Output( WHERE => 'DATABASE', NAV_ONE => 'Converting from an older Ikonboard', PRINT => $html);
}

sub d_mods {
    my ($obj, $db) = @_;

    my @ids =  $iB::CGI->param('f');

    my $base_dir = $INFO->{'DB_DIR'}.'member_profiles';

    # =tie a hash to the names index DB
    tie (my %N_INDEX, $AnyDBM_File::ISA[0], "$base_dir/MEMBER_NAME.idx", O_RDWR|O_CREAT, 0777);

    for my $m (@ids) {

        my $forum_id = $iB::IN{'FS_'.$m};
        
        for my $mds ( split(/, /, $iB::IN{'FM_'.$m}) ) {
            my $m_id = $N_INDEX{ $mds };
            next unless defined $m_id;
            $db->insert( TABLE  => 'forum_moderators',
                         VALUES => { FORUM_ID         => $forum_id,
                                     MEMBER_NAME      => $mds,
                                     MEMBER_ID        => $m_id,
                                     EDIT_POST        => 1,
                                     EDIT_TOPIC       => 1,
                                     DELTE_POST       => 1,
                                     DELETE_TOPIC     => 1,
                                     VIEW_IP          => 1,
                                     OPEN_TOPIC       => 1,
                                     CLOSE_TOPIC      => 1,
                                     MOVE_TOPIC       => 1,
                                     MASS_MOVE        => 1,
                                     MASS_PRUNE       => 1,
                                     MOVE_TOPIC       => 1,
                                     PIN_TOPIC        => 1,
                                     UNPIN_TOPIC      => 1,
                                     POST_Q           => 1,
                                     TOPIC_Q          => 1,
                                   },
                       );
        } # end for

    } # end other for :D

    untie (%N_INDEX);

    open LOCKFILE, ">".$INFO->{'DB_DIR'}.'Temp/mods.lockfile';
    print LOCKFILE "Done: ".time."\n";
    close LOCKFILE;
    $ADMIN->redirect( URL => "act=convert&CODE=one&skip=1", TEXT => 'Finishing up the moderators conversion process');
}


###################################################################
#
# TOPICS AND POSTS set-up
#
###################################################################


sub posts {
    my ($obj, $db) = @_;

    # Have we been here before?

    open TEMPFILE, $INFO->{'DB_DIR'}.'Temp/posts.lockfile';
    my @temp_data = <TEMPFILE>;
    close TEMPFILE;

    open DESTFILE, $INFO->{'DB_DIR'}.'Temp/dest_posts.lockfile';
    my @dest_data = <DESTFILE>;
    close DESTFILE;

    # Get the config file
    my $saved = $obj->_load_config;

    $ADMIN->Error( DB => $db, STD => $std, MSG => "Could not find the ib2 converter datafile")   unless $saved->{'LOCATION'};

    # Get iB2 forums as an array reference holding associative array references

    my $old_forums = $obj->_get_old_forums($saved);

    # Sort out the forums stuff

    my $forums   = $db->query( TABLE     => 'forum_info',
                               COLUMNS   => ['FORUM_NAME', 'FORUM_ID'],
                             ) || die $db->{'error'};

    my ($source, $destination);

    for my $of (@{$old_forums}) {
        next unless defined $of->{'t_ID'};
        my $text;
        if ( grep { $of->{'t_ID'} == $_ } @temp_data ) {
            $text = " [ Converted ]";
        }
        $source .= qq!<option value="$of->{'t_ID'}">$of->{'FORUM_NAME'}$text</option>!;
    }
    for my $rf (@{$forums}) {
        my $text;
        if ( grep { $rf->{'FORUM_ID'} == $_ } @dest_data ) {
            $text = " [ Converted ]";
        }
        $destination .= qq!<option value="$rf->{'FORUM_ID'}">$rf->{'FORUM_NAME'}$text</option>!;
    }

    my $html  = $SKIN->title( TITLE => 'Converting Ikonboard 2 to Ikonboard 3', TEXT => "Converting Topics and Posts Information" );
       $html .= $SKIN->begin_table();
       $html .= $SKIN->form_start();
       $html .= $SKIN->hidden_fields( { act   => 'convert',
                                        CODE  => 'd_posts'
                                      } );

    $html .= $SKIN->section_header( TITLE => "Please choose a source (Ikonboard 2) forum and the destination (Ikonboard 3) forum to convert into", TEXT => "For efficiency, only one forum at a time can be converted");

    $html .= qq~
         <table cellspacing='1' cellpadding='0' width='100%' align='center' border='0' bgcolor='#333366'>
             <tr>
             <td>
             <table cellspacing='1' cellpadding='2' width='100%' align='center' border='0'>
             <tr>
               <td bgcolor='#666699' width='50%'><font class='t' style='color:white;font-size:12px'><b>iB 2 Source Forum Name</b></font></td>
               <td bgcolor='#666699' width='50%'><font class='t' style='color:white;font-size:12px'><b>iB 3 Destination Forum Name</b></font></td>
             </tr>
             <tr>
               <td bgcolor='#FFFFFF'><font class='t'>Convert from...<select name='S'>$source</select></td>
               <td bgcolor='#FFFFFF'><font class='t'>Convert to...<select name='D'>$destination</select></td>
             </tr>
            <tr>
            <td colspan='6' align='center' bgcolor='#FFFFFF'><br><input type='submit' value='IMPORT POSTS AND TOPICS'></td>
            </tr>
            </table>
         </td>
        </tr>
      </table>
     </form>
    </td>
    </tr>
    </table>
    ~;
    $ADMIN->Output( WHERE => 'DATABASE', NAV_ONE => 'Converting from an older Ikonboard', PRINT => $html);
}


sub d_posts {
    my ($obj, $db) = @_;

    $ADMIN->Error( DB => $db, STD => $std, MSG => "No source forum selected!")        unless $iB::IN{'S'};
    $ADMIN->Error( DB => $db, STD => $std, MSG => "No destination forum selected!")   unless $iB::IN{'D'};

    # Get the config file
    my $saved = $obj->_load_config;

    $ADMIN->Error( DB => $db, STD => $std, MSG => "Could not find the ib2 converter datafile")   unless $saved->{'LOCATION'};

    # Make sure we have a valid start variable
    $iB::IN{'st'} ||= 0;

    # Get the end variable
    my $end = $iB::IN{'st'} + $saved->{'POST_COUNT'};

    # Get the topics list from the old Ikonboard

    open LISTFILE, $saved->{'LOCATION'}.'/forum'.$iB::IN{'S'}.'/list.cgi' or $ADMIN->Error( DB => $db, STD => $std, MSG => "Could not open the list file for source forum (forum $iB::IN{'S'}): $!");
    my @list = <LISTFILE>;
    close LISTFILE;

    my $m_total = scalar(@list);

    $end = $m_total if $end > $m_total;

    # =tie a hash to the names index DB
    my $base_dir = $INFO->{'DB_DIR'}.'member_profiles';
    tie (my %N_INDEX, $AnyDBM_File::ISA[0], "$base_dir/MEMBER_NAME.idx", O_RDWR|O_CREAT, 0777);


#     0       1               2              3            4            5           6             7             8             9
# $topicid $topictitle $topicdescription $threadstate $threadposts $threadviews $startedby $startedpostdate $lastposter $lastpostdate

#     0              1             2              3                  4             5          6
# $inmembername $intopictitle $postipaddress $inshowemoticons $inshowsignature $currenttime $inpost

    for my $entry (@list[$iB::IN{'st'} .. $end]) {
        chomp $entry;
        next unless $entry =~ m!|!;
        my @fields = split/\|/, $entry;
        next unless $fields[1];
        my $t_starter_id = $N_INDEX{ $fields[6] };
        my $t_lastposter = $N_INDEX{ $fields[8] };

        my $new_id = $db->insert( TABLE   => 'forum_topics',
                                  ID      => $iB::IN{'D'},
                                  VALUES  => { TOPIC_TITLE       => $fields[1],
                                               TOPIC_DESC        => $fields[2],
                                               TOPIC_STATE       => $fields[3],
                                               TOPIC_POSTS       => $fields[4],
                                               TOPIC_VIEWS       => $fields[5],
                                               TOPIC_STARTER     => $t_starter_id,
                                               TOPIC_STARTER_N   => $fields[6],
                                               TOPIC_START_DATE  => $fields[7],
                                               TOPIC_LAST_POSTER => $t_lastposter,
                                               TOPIC_LASTP_N     => $fields[8],
                                               TOPIC_LAST_DATE   => $fields[9],
                                               FORUM_ID          => $iB::IN{'D'},
                                               PIN_STATE         => 0,
                                               TOPIC_ICON        => 0,
                                               TOPIC_AUTHOR_TYPE => 1,
                                               APPROVED          => 1,
                                             },
                                 );
        # Grab the post...
        open POSTFILE, $saved->{'LOCATION'}.'/forum'.$iB::IN{'S'}.'/'.$fields[0].'.thd' or next;
        my @post = <POSTFILE>;
        close POSTFILE;

        for my $pants (@post) {
            chomp $pants;
            next unless $pants =~ m!|!;
            my @bits = split/\|/, $pants;
            next unless $bits[0];

            #$bits[4] = $bits[4] eq 'yes' ? 1 : 0;
            #$bits[3] = $bits[3] eq 'yes' ? 1 : 0;
            $bits[6] = $txt->Convert_for_db( TEXT    => $bits[6],
                                             SMILIES => $bits[3],
                                             IB_CODE => 1,
                                             HTML    => 0,
                                           );
            # Insert into post DB

            my $p_id = $db->insert(  TABLE  => 'forum_posts',
                                     DBID   => 'f'.$iB::IN{'D'},
                                     ID     => $new_id,
                                     VALUES => { AUTHOR        => $N_INDEX{ $bits[0] },
                                                 ENABLE_SIG    => $bits[4],
                                                 ENABLE_EMO    => $bits[3],
                                                 IP_ADDR       => $bits[2],
                                                 POST_DATE     => $bits[5],
                                                 POST_ICON     => 0,
                                                 POST          => $bits[6],
                                                 AUTHOR_TYPE   => 1,
                                                 QUEUED        => 0,
                                                 TOPIC_ID      => $new_id,
                                                 FORUM_ID      => $iB::IN{'D'},
                                                 ATTACH_ID     => undef,
                                                 ATTACH_HITS   => undef,
                                                 ATTACH_TYPE   => undef,
                                               },
                                  );

            # Insert into the search DB

            my $r_txt = $bits[6];
            $r_txt =~ s!<(.+?)>!!sg;
            $r_txt =~ s!\b($iB::INFO->{'SKIP_WORDS'})\b!!ig;
            $r_txt =~ s!  ! !g;
            $r_txt =~ s!&.+?;!!g;
            $r_txt = substr($r_txt, 0, $iB::INFO->{'MAX_CHARS'}) unless $iB::INFO->{'MAX_CHARS'} eq 'ALL' or $iB::INFO->{'MAX_CHARS'} eq '';
                 
            $db->insert( TABLE   => 'search_log',
                         ID      => $iB::IN{'D'},
                         VALUES  => { AUTHOR_ID    => $N_INDEX{ $bits[0] },
                                      MEMBER_NAME  => $bits[0],
                                      DATE         => $bits[5],
                                      TOPIC_TITLE  => $fields[1],
                                      POST         => $r_txt,
                                      FORUM_ID     => $iB::IN{'D'},
                                      TOPIC_ID     => $new_id,
                                      POST_ID      => $p_id,
                                      POSTER_IP    => $bits[2],
                                    },
                       );

        } # End of post insert

    } # End of topic insert

    untie (%N_INDEX);

    # What to do next..

    #increment the $end value to use as our start on the next invocation
    ++$end;
    # If we've processed the last of the topics, go to the "done" screen
    if ($end >= $m_total) {
        # Write to the lock files
        open TEMPFILE, ">>".$INFO->{'DB_DIR'}.'Temp/posts.lockfile';
        print TEMPFILE $iB::IN{'S'}."\n";
        close TEMPFILE;
    
        open DESTFILE, ">>".$INFO->{'DB_DIR'}.'Temp/dest_posts.lockfile';
        print DESTFILE $iB::IN{'D'}."\n";
        close DESTFILE;

        # Recount the forum...

        my $post_count = $db->count( TABLE  => 'forum_posts',
                                     DBID   => 'f'.$iB::IN{'D'}
                                   );
    
        my $topic_count = $db->count( TABLE => 'forum_topics',
                                      ID    => $iB::IN{'D'}
                                    );
    
        #XXX We need the replies only, so we substract the no. topics from the count
    
        $post_count -= $topic_count;

        # Get the details of the last poster..

        my $lp = $db->query( TABLE      => 'forum_topics',
                             ID         => $iB::IN{'D'},
                             SORT_KEY   => 'TOPIC_LAST_DATE',
                             SORT_BY    => 'Z-A',
                             RANGE      => "0 to 1",
                           );


    
        $db->update( TABLE  => 'forum_info',
                     KEY    => $iB::IN{'D'},
                     VALUES => { FORUM_TOPICS         => $topic_count,
                                 FORUM_POSTS          => $post_count,
                                 FORUM_LAST_POST      => $lp->[0]->{'TOPIC_LAST_DATE'},
                                 FORUM_LAST_POSTER    => $lp->[0]->{'TOPIC_LAST_POSTER'},
                                 FORUM_LAST_POSTER_N  => $lp->[0]->{'TOPIC_LASTP_N'},
                                 L_TOPIC_TITLE        => $lp->[0]->{'TOPIC_TITLE'},
                                 L_TOPIC_ID           => $lp->[0]->{'TOPIC_ID'},
                               }
                   );

       $ADMIN->redirect( URL => "act=convert&CODE=posts", TEXT => 'Finishing up the forum convertion process');
    } else {
       $ADMIN->redirect( URL => "act=convert&CODE=d_posts&st=$end&S=$iB::IN{'S'}&D=$iB::IN{'D'}", TEXT => 'Processing the next batch...');
    }
}












sub process {
    my ($obj, $db) = @_;

    my $CodeNo = $iB::IN{'CODE'};

    my %Mode = ( 'one'        => \&step_one,
                 'members'    => \&members,
                 'forums'     => \&forums,
                 'd_forums'   => \&d_forums,
                 'posts'      => \&posts,
                 'd_posts'    => \&d_posts,
                 'mods'       => \&mods,
                 'd_mods'     => \&d_mods,
                 'cats'       => \&cats,
                 'd_cats'     => \&do_cats,
               );
    $Mode{$CodeNo} ? $Mode{$CodeNo}->($obj,$db) : splash($obj,$db);
} 




###################################################################
#
# INTERNAL SUBS
#
###################################################################

sub _load_db_cfg ($$) {
    my ($obj, $cfg) = @_;
    do $INFO->{'DB_DIR'}.'config/'.$cfg.'.cfg' or die "Cannot open file : $cfg.cfg ($!)";

    $obj->{'cur_table'}  = $IMPORT::STRING->{'TABLE'};
    $obj->{'cur_p_key'}  = $IMPORT::STRING->{'P_KEY'};
    $obj->{'cur_method'} = $IMPORT::STRING->{'MTD'};
    $obj->{'cur_update'} = $IMPORT::STRING->{'UPDATE'} || 'bottom';
    $obj->{'all_cols'}   = $IMPORT::COLS;
    $obj->{'total_cols'} = 0;
    $obj->{'col_name'}   = [];
    foreach (sort { $obj->{'all_cols'}->{$a}[0] <=> $obj->{'all_cols'}->{$b}[0] } keys %{$obj->{'all_cols'}} ) {
        push @{$obj->{'col_name'}}, $_;
        ++$obj->{'total_cols'};
    }
    $obj->{'total_cols'}--;
}

sub _encode_record {
    my ($obj, $values) = @_;
    my ($return, $cnt);
    for my $i (0 .. $obj->{'total_cols'}) {
        $values->{$obj->{'col_name'}->[$i]}  =~ s!^\s+!!g;
        $values->{$obj->{'col_name'}->[$i]}  =~ s!\s+$!!g;
        $values->{$obj->{'col_name'}->[$i]}  =~ s!\r!!g;
        $values->{$obj->{'col_name'}->[$i]}  =~ s!\n!\\n!g;
        $return .= $values->{$obj->{'col_name'}->[$i]}."|^|";
    }
  $return =~ s!\Q|^|\E$!!;
  return $return;
}


sub _get_old_forums {
    my ($obj, $saved) = @_;

    open FORUMDATA, $saved->{'LOCATION'}.'/data/allforums.cgi' or return {};
    my @forums = <FORUMDATA>;
    close FORUMDATA;

    my @temp;

    for my $f (@forums) {
        chomp $f;
        next unless $f;
        my @t_f = split(/\|/, $f);
        my $html = $t_f[6] eq 'off' ? 0 : 1;
        my $code = $t_f[7] eq 'off' ? 0 : 1;

        $temp[ $t_f[0] ] = { FORUM_NAME  => $t_f[3],
                             FORUM_DESC  => $t_f[4],
                             FORUM_IBC   => $code,
                             FORUM_HTML  => $html,
                             t_MODS      => $t_f[5],     # Get the mods, complete with commas
                             t_ID        => $t_f[0],
                             t_CAT       => $t_f[1],
                           };
    }

    return \@temp;
}


sub _get_old_cats {
    my ($obj, $saved) = @_;

    open FORUMDATA, $saved->{'LOCATION'}.'/data/allforums.cgi' or return {};
    my @forums = <FORUMDATA>;
    close FORUMDATA;

    my @temp;
    my $this_cat;
    my $last_cat;

    for my $f (@forums) {
        chomp $f;
        next unless $f;
        my @t_cat = split(/\|/, $f);
        next unless defined $t_cat[1] && $t_cat[2];
        $this_cat = $t_cat[1];
        if ($this_cat ne $last_cat) {
            $temp[ $t_cat[2] ] = { CAT_NAME => $t_cat[1],
                                   CAT_POS  => $t_cat[2],
                                 };
            $last_cat = $t_cat[1];
        }
    }

    return \@temp;
}




sub _split_mem {
    my ($obj, $file_name, $saved) = @_;

    open MEMBER, $saved->{'LOCATION'}.'/members/'.$file_name or return {};
    my $mem_data = <MEMBER>;
    close MEMBER;

    chomp $mem_data;

    my @mem_array = split(/\|/, $mem_data);

    my $return = { MEMBER_NAME     => $mem_array[0],
                   MEMBER_PASSWORD => $mem_array[1],
                   MEMBER_TITLE    => '',
                   MEMBER_LEVEL    => 1,
                   t_CODE          => $mem_array[3],
                   MEMBER_POSTS    => $mem_array[4],
                   MEMBER_EMAIL    => lc($mem_array[5]),
                   HIDE_EMAIL      => $mem_array[6] || 0,
                   MEMBER_IP       => $mem_array[7],
                   WEBSITE         => $mem_array[8],
                   AOLNAME         => $mem_array[9],
                   ICQNUMBER       => $mem_array[10],
                   LOCATION        => $mem_array[11],
                   INTERESTS       => $mem_array[12],
                   MEMBER_JOINED   => $mem_array[13],
                   SIGNATURE       => $mem_array[15],
                   TIME_ADJUST     => $mem_array[16],
                   MEMBER_AVATAR   => $mem_array[18],
                   AVATAR_DIMS     => '32x32',
                   ALLOW_ADMIN_EMAILS => 1,
                   LAST_UPDATE     => time,
                   ALLOW_POST      => 1,
                   VIEW_SIGS       => 1,
                   VIEW_IMG        => 1,
                   VIEW_AVS        => 1,
                   EMAIL_FULL_POST => 0,
                   PM_REMINDER     => '1&1',
                 };

    if ($return->{'t_CODE'} eq 'ad') {
        $return->{'MEMBER_GROUP'} = $saved->{'ADMIN_GROUP'};
    } elsif
       ($return->{'t_CODE'} eq 'mo') {
        $return->{'MEMBER_GROUP'} = $saved->{'MOD_GROUP'};
    } else {
        $return->{'MEMBER_GROUP'} = $saved->{'MEMBER_GROUP'};
    }
    delete $return->{'t_CODE'};

    # Tidy up the members avatar
    # Thanks to 'joshdw1' for his assistance, time and server to fix this up.

    if ($return->{'MEMBER_AVATAR'} =~ /^\w+$/) {
        $return->{'MEMBER_AVATAR'} .= '.gif';
    }

    $return->{'LOCATION'}     = $txt->Convert_for_db( TEXT    => $return->{'LOCATION'},
                                                      SMILIES => 0,
                                                      IB_CODE => 0,
                                                      HTML    => 0,
                                                    );
    $return->{'INTERESTS'}    = $txt->Convert_for_db( TEXT    => $return->{'INTERESTS'},
                                                      SMILIES => 0,
                                                      IB_CODE => 0,
                                                      HTML    => 0,
                                                    );
    $return->{'SIGNATURE'}    = $txt->Convert_for_db( TEXT    => $return->{'SIGNATURE'},
                                                      SMILIES => $iB::INFO->{'SIG_ALLOW_EMOTICONS'},                                                              SMILIES => 0,
                                                      IB_CODE => $iB::INFO->{'SIG_ALLOW_IBC'},
                                                      HTML    => $iB::INFO->{'SIG_ALLOW_HTML'},
                                                      SIG     => 1
                                                    );

    return $return;
}


sub _load_config {
    my $obj = shift;
    return {} unless (-e $INFO->{'DB_DIR'}."Temp/ib2.data");
    my $config = {};
    open CONFIGFILE, $INFO->{'DB_DIR'}."Temp/ib2.data";
    while (<CONFIGFILE>) {
        next if /^\s+$/;
        my ($k, $v) = (/^(\w+)\s=\s(.*)$/);
        $config->{$k} = $v;
    }
    close CONFIGFILE;
    return $config;
}

sub _check_convert {
    my $obj = shift;
    my $return = {};
    $return->{'MEMBERS'} = (-e $INFO->{'DB_DIR'}."Temp/members.lockfile")    ? 1 : 0;
    $return->{'CATS'}    = (-e $INFO->{'DB_DIR'}."Temp/categories.lockfile") ? 1 : 0;
    $return->{'FORUMS'}  = (-e $INFO->{'DB_DIR'}."Temp/forums.lockfile")     ? 1 : 0;
    $return->{'MODS'}    = (-e $INFO->{'DB_DIR'}."Temp/mods.lockfile")       ? 1 : 0;
    $return->{'POSTS'}   = (-e $INFO->{'DB_DIR'}."Temp/posts.lockfile")      ? 1 : 0;
    return $return;
}

sub _write_config {
    my ($obj, $config) = @_;
    open CONFIGFILE, ">".$INFO->{'DB_DIR'}."Temp/ib2.data";
    for my $k ( sort { $a cmp $b } keys %{$config} ) {
        print CONFIGFILE "$k = $config->{$k}\n";
    }
    close CONFIGFILE;
}


1;
